(function($) {
    'use strict';

    let lastTagIds = [];
    let lastCategoryIds = [];
    let debounceTimer = null;

    $(document).ready(function() {
        initWppAdiFeed();
    });

    function isClassicEditor() {
        // Detecta editor clássico: verifica se há elementos específicos do editor clássico
        // O WordPress pode carregar wp.data mesmo com editor clássico ativo, então verificamos o DOM
        return $('#post-body-content').length > 0 || ($('#postdivrich').length > 0 && $('#content').length > 0);
    }

    function getClassicTagSlugs() {
        let slugs = [];
        // Tags aparecem na checklist como texto após o botão de remover
        $('.tagchecklist li').each(function() {
            // Pegar apenas os nós de texto diretos do <li> (ignorando elementos HTML)
            let tagSlug = '';
            $(this).contents().filter(function() {
                return this.nodeType === 3; // Node.TEXT_NODE
            }).each(function() {
                let text = this.nodeValue.trim();
                if (text && text !== '×' && !text.includes('Remover termo:')) {
                    tagSlug = text;
                }
            });
            if (tagSlug) {
                slugs.push(tagSlug);
            }
        });
        console.log('[WPP ADI] Tag slugs encontrados:', slugs);
        return slugs;
    }

    function getClassicCategories() {
        let cats = [];
        $("input[type=checkbox][name^='post_category']:checked").each(function() {
            cats.push(parseInt($(this).val()));
        });
        console.log('[WPP ADI] Categorias encontradas:', cats);
        return cats;
    }

    function initWppAdiFeed() {
        loadValidTags();
        setupPostMetaBox();
        if (isClassicEditor()) {
            setupClassicEditorValidation();
        } else {
            setupTagValidation();
            setupCategoryClickValidation();
        }
    }

    function setupClassicEditorValidation() {
        console.log('[WPP ADI] Configurando validação para Editor Clássico');
        
        // Validação combinada
        async function validateClassicEditor() {
            const tagSlugs = getClassicTagSlugs();
            const categoryIds = getClassicCategories();
            
            // Validar tags diretamente (já temos os slugs)
            const validTagSlugs = tagSlugs.filter(slug => 
                window.wppAdiFeedValidTags && window.wppAdiFeedValidTags.includes(slug)
            );
            
            // Para categorias, buscar slugs via API e combinar com tags válidas
            if (categoryIds.length > 0) {
                await validateCurrentTags([], categoryIds, validTagSlugs);
            } else {
                // Só tags, mostrar resultado direto
                updateStatusDisplay(validTagSlugs);
            }
        }
        
        // Observar mudanças na checklist de tags
        const tagChecklist = document.querySelector('.tagchecklist');
        if (tagChecklist) {
            const observer = new MutationObserver(function() {
                setTimeout(validateClassicEditor, 300);
            });
            observer.observe(tagChecklist, { childList: true, subtree: true });
        }
        
        // Valida ao alterar tags
        $(document).on('change', "input[name='tax_input[post_tag][]'], #new-tag-post_tag", function() {
            console.log('[WPP ADI] Tag alterada');
            setTimeout(validateClassicEditor, 300);
        });
        
        // Valida ao alterar categorias
        $(document).on('change', "input[type=checkbox][name^='post_category']", function() {
            console.log('[WPP ADI] Categoria alterada');
            setTimeout(validateClassicEditor, 300);
        });
    }
    // Força validação ao clicar em uma categoria no editor Gutenberg
    function setupCategoryClickValidation() {
        // Espera o DOM do painel de categorias do Gutenberg
        const interval = setInterval(() => {
            // Gutenberg: painel de categorias
            const categoryPanel = document.querySelector('[aria-label="Categorias"]');
            if (categoryPanel) {
                categoryPanel.addEventListener('click', function() {
                    // Aguarda atualização do estado do editor
                    setTimeout(() => {
                        if (typeof wp !== 'undefined' && wp.data) {
                            const tags = wp.data.select('core/editor').getEditedPostAttribute('tags') || [];
                            const categories = wp.data.select('core/editor').getEditedPostAttribute('categories') || [];
                            validateCurrentTags(tags, categories);
                        }
                    }, 300);
                });
                clearInterval(interval);
            }
        }, 1000);
    }

    function loadValidTags() {
        showLoadingIndicator();

        $.ajax({
            url: wppAdiFeed.ajax_url,
            type: 'POST',
            data: {
                action: 'load_gateway_tags',
                nonce: wppAdiFeed.nonce
            },
            success: function(response) {
                if (response.success) {
                    window.wppAdiFeedValidTags = response.data;
                    updateTagsDisplay(response.data);

                    // Verificar qual editor está ativo e validar
                    if (typeof wp !== 'undefined' && wp.data && wp.data.select && typeof wp.data.select === 'function' && wp.data.select('core/editor')) {
                        // Gutenberg detectado
                        console.log('[WPP ADI] Gutenberg detectado');
                        const editor = wp.data.select('core/editor');
                        const currentTags = editor.getEditedPostAttribute("tags") || [];
                        const currentCategories = editor.getEditedPostAttribute("categories") || [];
                        validateCurrentTags(currentTags, currentCategories);
                    } else if (isClassicEditor()) {
                        // Editor clássico: validar após carregamento das tags
                        setTimeout(function() {
                            console.log('[WPP ADI] Iniciando validação inicial no editor clássico');
                            const tagSlugs = getClassicTagSlugs();
                            const categoryIds = getClassicCategories();
                            const validTagSlugs = tagSlugs.filter(slug => window.wppAdiFeedValidTags && window.wppAdiFeedValidTags.includes(slug));
                            if (categoryIds.length > 0) {
                                validateCurrentTags([], categoryIds, validTagSlugs);
                            } else {
                                updateStatusDisplay(validTagSlugs);
                            }
                        }, 500);
                    }
                } else {
                    showError('Erro ao carregar tags: ' + response.data);
                }
            },
            error: function() {
                showError('Erro de conexão ao carregar tags');
            },
            complete: function() {
                hideLoadingIndicator();
            }
        });
    }

    function setupTagValidation() {
        if (!wp || !wp.data || !wp.data.subscribe) {
            console.warn('wp.data.subscribe não está disponível.');
            return;
        }

        wp.data.subscribe(() => {
            const tags = wp.data.select('core/editor').getEditedPostAttribute('tags') || [];
            const categories = wp.data.select('core/editor').getEditedPostAttribute('categories') || [];

            const changed = JSON.stringify(tags) !== JSON.stringify(lastTagIds) ||
                            JSON.stringify(categories) !== JSON.stringify(lastCategoryIds);
            if (changed) {
                lastTagIds = tags;
                lastCategoryIds = categories;
                
                // Debounce: aguardar 500ms após última mudança antes de validar
                if (debounceTimer) {
                    clearTimeout(debounceTimer);
                }
                debounceTimer = setTimeout(() => {
                    validateCurrentTags(tags, categories);
                }, 500);
            }
        });
    }

    async function validateCurrentTags(tagIds = [], categoryIds = [], preValidatedSlugs = []) {
        console.log('[WPP ADI] Validando tags:', tagIds, 'categorias:', categoryIds, 'slugs pré-validados:', preValidatedSlugs);
        
        if (!window.wppAdiFeedValidTags) {
            console.warn('[WPP ADI] Tags válidas ainda não carregadas');
            return;
        }

        const validTags = window.wppAdiFeedValidTags;
        const slugs = [...preValidatedSlugs];

        // Função auxiliar para buscar slug de uma taxonomia
        async function getSlugFromId(taxonomy, id) {
            try {
                // Busca pública: context=view
                const apiUrl = `/wp-json/wp/v2/${taxonomy === 'category' ? 'categories' : 'tags'}/${id}?context=view&_locale=user`;
                const response = await fetch(apiUrl, { credentials: 'same-origin' });
                if (!response.ok) throw response;
                const term = await response.json();
                return term?.slug || null;
            } catch (error) {
                return null;
            }
        }

        // Buscar slugs via REST API para IDs fornecidos
        for (const id of tagIds) {
            const slug = await getSlugFromId('post_tag', id);
            if (slug && validTags.includes(slug)) {
                slugs.push(slug);
            }
        }
        for (const id of categoryIds) {
            const slug = await getSlugFromId('category', id);
            if (slug && validTags.includes(slug)) {
                slugs.push(slug);
            }
        }
        console.log('[WPP ADI] Slugs válidos finais:', slugs);
        updateStatusDisplay(slugs);
    }
    

    function setupPostMetaBox() {
        if (!$('#wpp-adi-feed-status').length) {
            addStatusMetaBox();
        }
    }

    function addStatusMetaBox() {
        const metaBox = `
            <div id="wpp-adi-feed-status" class="postbox">
                <div class="postbox-header">
                    <h2 class="hndle">Relatório ADI</h2>
                </div>
                <div class="inside">
                    <div id="wpp-adi-feed-content">
                        <p id="wpp-adi-feed-loading">${wppAdiFeed.loading_text}</p>
                        <div id="wpp-adi-feed-info" style="display: none;">
                            <p><strong>Status:</strong> <span id="wpp-adi-feed-status-text">Verificando...</span></p>
                        </div>
                    </div>
                </div>
            </div>
        `;
        // Se for editor clássico, inserir logo após o widget de tags
        if (isClassicEditor() && $('#tagsdiv-post_tag').length) {
            $('#tagsdiv-post_tag').after(metaBox);
        } else {
            // Gutenberg ou fallback
            $('#side-sortables').prepend(metaBox);
        }
    }

    function updateTagsDisplay(tags) {
        const tagsList = $('#wpp-adi-feed-valid-tags');
        tagsList.empty();

        if (tags && tags.length > 0) {
            tags.slice(0, 10).forEach(tag => {
                tagsList.append('<li>' + escapeHtml(tag) + '</li>');
            });

            if (tags.length > 10) {
                tagsList.append('<li><em>... e mais ' + (tags.length - 10) + ' tags</em></li>');
            }
        } else {
            tagsList.append('<li><em>Nenhuma tag válida encontrada</em></li>');
        }

        $('#wpp-adi-feed-info').show();
    }

    function updateStatusDisplay(validTerms) {
        const statusText = $('#wpp-adi-feed-status-text');
        if (validTerms.length > 0) {
            statusText.html('<span style="color: green;">✓ Post será registrado</span><br>' +
                            '<small>Tags/categorias válidas: ' + validTerms.join(', ') + '</small>');
        } else {
            statusText.html('<span style="color: orange;">⚠ Post não será registrado</span><br>' +
                            '<small>Nenhuma tag ou categoria válida encontrada</small>');
        }
        // Garantir que o widget de info seja exibido
        $('#wpp-adi-feed-loading').hide();
        $('#wpp-adi-feed-info').show();
    }

    function showLoadingIndicator() {
        $('#wpp-adi-feed-loading').show();
        $('#wpp-adi-feed-info').hide();
    }

    function hideLoadingIndicator() {
        $('#wpp-adi-feed-loading').hide();
    }

    function showError(message) {
        const errorHtml = '<div class="notice notice-error"><p>' + escapeHtml(message) + '</p></div>';
        $('#wpp-adi-feed-content').prepend(errorHtml);
    }

    function escapeHtml(text) {
        const map = {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#039;'
        };
        return text.replace(/[&<>"']/g, function(m) { return map[m]; });
    }

    // Adiciona notificação no salvamento
    $(document).on('click', '#publish, #save-post', function() {
        if (!window.wppAdiFeedValidTags) return;

        const tags = lastTagIds || [];
        const categories = lastCategoryIds || [];
        const allTerms = [...tags, ...categories].map(String);

        const hasValid = allTerms.some(term => window.wppAdiFeedValidTags.includes(term));

        if (hasValid) {
            setTimeout(function() {
                if ($('.notice-success').length > 0) {
                    $('.notice-success').first().append(
                        '<p><small>✓ Post registrado no WPP ADI Feed</small></p>'
                    );
                }
            }, 1000);
        }
    });

})(jQuery);
