<?php

/**
 * Classe para configurações administrativas
 */

if (!defined('ABSPATH')) {
    exit;
}

class WPP_ADI_Feed_Admin_Settings
{

    private $settings;
    private $api_client;

    public function __construct()
    {
        $this->settings = get_option('wpp_adi_feed_settings', array());

        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'settings_init'));
        add_action('admin_post_wpp_adi_feed_test_connection', array($this, 'test_connection'));
        add_action('admin_post_wpp_adi_feed_clear_cache', array($this, 'clear_cache'));
        add_action('admin_notices', array($this, 'check_schedule_status_notice'));
    }

    /**
     * Obter instância do API client (lazy loading)
     */
    private function get_api_client()
    {
        if (!$this->api_client && class_exists('WPP_ADI_Feed_API_Client')) {
            $this->api_client = new WPP_ADI_Feed_API_Client();
        }
        return $this->api_client;
    }

    /**
     * Adicionar menu administrativo
     */
    public function add_admin_menu()
    {
        add_options_page(
            __('ADI Feed Configurações', 'wpp-adi-feed'),
            __('ADI Feed Config', 'wpp-adi-feed'),
            'manage_options',
            'wpp-adi-feed',
            array($this, 'settings_page')
        );
    }

    /**
     * Inicializar configurações
     */
    public function settings_init()
    {
        register_setting('wpp_adi_feed_settings', 'wpp_adi_feed_settings', array($this, 'sanitize_settings'));

        add_settings_section(
            'wpp_adi_feed_api_section',
            __('Configurações da API', 'wpp-adi-feed'),
            array($this, 'api_section_callback'),
            'wpp_adi_feed_settings'
        );

        add_settings_field(
            'api_base_url',
            __('URL Base da API', 'wpp-adi-feed'),
            array($this, 'api_base_url_callback'),
            'wpp_adi_feed_settings',
            'wpp_adi_feed_api_section'
        );

        add_settings_field(
            'cdn_script_url',
            __('URL do Script CDN', 'wpp-adi-feed'),
            array($this, 'cdn_script_url_callback'),
            'wpp_adi_feed_settings',
            'wpp_adi_feed_api_section'
        );

        add_settings_field(
            'request_timeout',
            __('Timeout das Requisições (segundos)', 'wpp-adi-feed'),
            array($this, 'request_timeout_callback'),
            'wpp_adi_feed_settings',
            'wpp_adi_feed_api_section'
        );

        add_settings_field(
            'cache_duration',
            __('Duração do Cache (segundos)', 'wpp-adi-feed'),
            array($this, 'cache_duration_callback'),
            'wpp_adi_feed_settings',
            'wpp_adi_feed_api_section'
        );

        add_settings_field(
            'auto_register',
            __('Registro Automático', 'wpp-adi-feed'),
            array($this, 'auto_register_callback'),
            'wpp_adi_feed_settings',
            'wpp_adi_feed_api_section'
        );

        add_settings_field(
            'log_errors',
            __('Log de Erros', 'wpp-adi-feed'),
            array($this, 'log_errors_callback'),
            'wpp_adi_feed_settings',
            'wpp_adi_feed_api_section'
        );

        add_settings_field(
            'log_retention_days',
            __('Dias para manter os logs', 'wpp-adi-feed'),
            array($this, 'log_retention_days_callback'),
            'wpp_adi_feed_settings',
            'wpp_adi_feed_api_section'
        );

        add_action('admin_post_toggle_schedule', array($this, 'toggle_schedule_handler'));
    }

    /**
     * Página de configurações
     */
    public function settings_page()
    {
?>
        <div class="wrap wpp-adi-feed-settings">
            <style>
                .wpp-adi-feed-settings .top-bar {
                    display: flex;
                    align-items: center;
                    justify-content: space-between;
                    gap: 16px;
                    margin-bottom: 24px;
                    padding: 12px 18px;
                    background: #f6f7f7;
                    border-left: 2px solid #2271b1;
                    box-shadow: 0 1px 2px rgba(0, 0, 0, 0.04);
                }

                .wpp-adi-feed-settings .top-bar .version-badge {
                    background: #2271b1;
                    color: #fff;
                    padding: 4px 10px;
                    border-radius: 10px;
                    font-size: 11px;
                    font-weight: 600;
                }

                .wpp-adi-feed-settings .top-bar .actions {
                    display: flex;
                    gap: 8px;
                }

                .wpp-adi-feed-settings .top-bar h1 {
                    font-size: 22px;
                    font-weight: 700;
                    margin: 0;
                }

                @media (max-width: 600px) {
                    .wpp-adi-feed-settings .top-bar {
                        flex-direction: column;
                        align-items: flex-start;
                        gap: 8px;
                        padding: 8px 8px;
                    }

                    .wpp-adi-feed-settings .top-bar h1 {
                        font-size: 16px;
                    }
                }

                .wpp-adi-feed-settings .main-content {
                    display: grid;
                    grid-template-columns: 2fr 1fr;
                    gap: 24px;
                }

                .wpp-adi-feed-settings .form-panel {
                    background: #fff;
                    padding: 24px;
                    box-shadow: 0 1px 3px rgba(0, 0, 0, 0.07);
                    border-radius: 6px;
                }

                .wpp-adi-feed-settings .form-panel h2 {
                    font-size: 20px;
                    font-weight: 600;
                    margin-bottom: 18px;
                }

                .wpp-adi-feed-settings .tools {
                    display: flex;
                    gap: 16px;
                    margin-top: 18px;
                }

                .wpp-adi-feed-settings .tool-card {
                    flex: 1;
                    background: #f9f9f9;
                    border: 1px solid #ddd;
                    border-radius: 4px;
                    padding: 16px;
                    text-align: center;
                }

                .wpp-adi-feed-settings .tool-card .button {
                    width: 100%;
                    margin-top: 10px;
                }

                .wpp-adi-feed-settings .status-panel {
                    background: #fff;
                    padding: 24px;
                    box-shadow: 0 1px 3px rgba(0, 0, 0, 0.07);
                    border-radius: 6px;
                }

                .wpp-adi-feed-settings .status-panel h2 {
                    font-size: 18px;
                    font-weight: 600;
                    margin-bottom: 16px;
                }

                .wpp-adi-feed-settings .status-card {
                    padding: 14px 0;
                    border-bottom: 1px solid #f0f0f0;
                }

                .wpp-adi-feed-settings .status-label {
                    font-weight: 600;
                    margin-bottom: 4px;
                    display: block;
                }

                .wpp-adi-feed-settings .status-value {
                    color: #666;
                }

                .wpp-adi-feed-settings .status-success {
                    color: #00a32a;
                }

                .wpp-adi-feed-settings .status-error {
                    color: #d63638;
                }

                .wpp-adi-feed-settings .status-warning {
                    color: #dba617;
                }

                @media (max-width: 1280px) {
                    .wpp-adi-feed-settings .main-content {
                        grid-template-columns: 1fr;
                    }
                }

                .wpp-adi-feed-settings .schedule-panel {
                    background: #fff;
                    padding: 24px;
                    box-shadow: 0 1px 3px rgba(0, 0, 0, 0.07);
                    border-radius: 6px;
                    margin-top: 24px;
                }

                .wpp-adi-feed-settings .schedule-panel h2 {
                    font-size: 18px;
                    font-weight: 600;
                    margin-bottom: 16px;
                }

                .wpp-adi-feed-settings .schedule-status {
                    font-weight: 600;
                    margin-bottom: 8px;
                }

                .wpp-adi-feed-settings .schedule-next {
                    font-size: 13px;
                    color: #666;
                    margin-bottom: 12px;
                }
            </style>

            <div class="top-bar-">
                <div style=" align-items: center; gap: 12px;">
                    <h1>WPP ADI Feed</h1>
                </div>
            </div>

            <div class="top-bar">
                <div style="display: flex; align-items: center; gap: 12px;">
                    <span class="version-badge">v<?php echo WPP_ADI_FEED_VERSION; ?></span>
                </div>
                <div class="actions">
                    <form method="post" action="<?php echo admin_url('admin-post.php'); ?>" style="margin: 0;">
                        <?php wp_nonce_field('wpp_adi_feed_test_connection', 'wpp_adi_feed_nonce'); ?>
                        <input type="hidden" name="action" value="wpp_adi_feed_test_connection">
                        <button type="submit" class="button button-secondary">Testar Conexão</button>
                    </form>
                    <form method="post" action="<?php echo admin_url('admin-post.php'); ?>" style="margin: 0;">
                        <?php wp_nonce_field('wpp_adi_feed_clear_cache', 'wpp_adi_feed_nonce'); ?>
                        <input type="hidden" name="action" value="wpp_adi_feed_clear_cache">
                        <button type="submit" class="button button-secondary">Limpar Cache</button>
                    </form>
                </div>
            </div>
            <?php
            // Mensagens de feedback
            if (isset($_GET['message'])) {
                $message = sanitize_text_field($_GET['message']);
                switch ($message) {
                    case 'connection_success':
                        echo '<div class="notice notice-success is-dismissible"><p><strong>✓</strong> ' . __('Conexão com a API testada com sucesso!', 'wpp-adi-feed') . '</p></div>';
                        break;
                    case 'connection_failed':
                        echo '<div class="notice notice-error is-dismissible"><p><strong>✗</strong> ' . __('Falha na conexão com a API.', 'wpp-adi-feed') . '</p></div>';
                        break;
                    case 'cache_cleared':
                        echo '<div class="notice notice-success is-dismissible"><p><strong>✓</strong> ' . __('Cache limpo com sucesso!', 'wpp-adi-feed') . '</p></div>';
                        break;
                    case 'manual_log_cleanup_done':
                        $deleted_count = isset($_GET['deleted']) ? intval($_GET['deleted']) : 0;
                        $message = $deleted_count > 0
                            ? sprintf(__('Limpeza de logs executada com sucesso! %d logs foram removidos.', 'wpp-adi-feed'), $deleted_count)
                            : __('Limpeza de logs executada com sucesso! Nenhum log foi removido.', 'wpp-adi-feed');
                        echo '<div class="notice notice-success is-dismissible"><p><strong>✓</strong> ' . $message . '</p></div>';
                        break;
                }
            }
            ?>

            <div class="main-content">
                <div class="form-panel">
                    <h2><?php _e('Configurações', 'wpp-adi-feed'); ?></h2>
                    <form action="options.php" method="post">
                        <?php
                        settings_fields('wpp_adi_feed_settings');
                        do_settings_sections('wpp_adi_feed_settings');
                        submit_button(__('Salvar Configurações', 'wpp-adi-feed'), 'primary');
                        ?>
                    </form>

                    <div class="schedule-panel">
                        <h2><?php _e('Schedule de Limpeza de Logs', 'wpp-adi-feed'); ?></h2>
                        <?php
                        $is_active = wp_next_scheduled('wpp_adi_feed_daily_log_cleanup');
                        if ($is_active) {
                            $next_run = wp_next_scheduled('wpp_adi_feed_daily_log_cleanup');
                            echo '<div class="schedule-status" style="color: #00a32a;">✓ Ativo</div>';
                            echo '<div class="schedule-next">Próxima execução: ' . date_i18n('d/m/Y H:i', $next_run) . '</div>';
                            echo '<a href="' . admin_url('admin-post.php?action=toggle_schedule&status=deactivate') . '" class="button button-secondary">Desativar</a>';
                            // Botão para executar limpeza manualmente
                        ?>
                            <form method="post" action="<?php echo admin_url('admin-post.php'); ?>" style="display:inline-block; margin-left:12px;">
                                <?php wp_nonce_field('wpp_adi_feed_manual_log_cleanup', 'wpp_adi_feed_nonce'); ?>
                                <input type="hidden" name="action" value="wpp_adi_feed_manual_log_cleanup">
                                <button type="submit" class="button button-primary" style="margin-top:10px;">
                                    <?php _e('Executar Limpeza de Logs Agora', 'wpp-adi-feed'); ?>
                                </button>
                            </form>
                            <p style="margin-top: 10px; font-size: 12px; color: #666;">
                                <em>Se o botão acima não funcionar, <a href="<?php echo wp_nonce_url(admin_url('admin-post.php?action=wpp_adi_feed_manual_log_cleanup'), 'wpp_adi_feed_manual_log_cleanup', 'wpp_adi_feed_nonce'); ?>" style="color: #2271b1;">clique aqui para teste alternativo</a>.</em>
                            </p>
                        <?php
                        } else {
                            echo '<div class="schedule-status" style="color: #dba617;">⚠ Inativo</div>';
                            echo '<div class="schedule-next">Nenhuma execução agendada</div>';
                            echo '<a href="' . admin_url('admin-post.php?action=toggle_schedule&status=activate') . '" class="button button-primary">Ativar</a>';
                        }
                        ?>
                    </div>
                </div>
                <div class="status-panel">
                    <h2><?php _e('Status do Sistema', 'wpp-adi-feed'); ?></h2>
                    <?php $this->show_status_info(); ?>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Mostrar informações de status
     */
    private function show_status_info()
    {
        $api_client = $this->get_api_client();

        if (!$api_client) {
        ?>
            <div class="status-card status-error">
                <span class="status-label">⚠️ <?php _e('Erro', 'wpp-adi-feed'); ?></span>
                <span class="status-value"><?php _e('Cliente API não disponível', 'wpp-adi-feed'); ?></span>
            </div>
        <?php
            return;
        }

        $valid_tags = $api_client->get_valid_tags();
        $cache_info = get_transient('wpp_adi_feed_valid_tags');
        $is_schedule_active = wp_next_scheduled('wpp_adi_feed_daily_log_cleanup');

        ?>
        <div class="status-card status-success">
            <span class="status-label"><?php _e('Conexão com API', 'wpp-adi-feed'); ?></span>
            <?php if ($valid_tags !== false): ?>
                <span class="status-value" style="color: #00a32a; font-weight: 600;">✓ <?php _e('Conectado', 'wpp-adi-feed'); ?></span>
            <?php else: ?>
                <span class="status-value" style="color: #d63638; font-weight: 600;">✗ <?php _e('Desconectado', 'wpp-adi-feed'); ?></span>
            <?php endif; ?>
        </div>

        <div class="status-card status-warning">
            <span class="status-label"><?php _e('Cache de Tags', 'wpp-adi-feed'); ?></span>
            <?php if ($cache_info !== false): ?>
                <span class="status-value" style="color: #00a32a; font-weight: 600;">✓ <?php _e('Ativo', 'wpp-adi-feed'); ?></span>
            <?php else: ?>
                <span class="status-value" style="color: #dba617; font-weight: 600;">⚠ <?php _e('Vazio', 'wpp-adi-feed'); ?></span>
            <?php endif; ?>
        </div>

        <?php if ($valid_tags !== false): ?>
            <div class="status-card status-success">
                <span class="status-label"><?php _e('Tags Válidas', 'wpp-adi-feed'); ?></span>
                <span class="status-value">
                    <strong><?php echo count($valid_tags); ?></strong> tags carregadas
                    <details style="margin-top: 8px;">
                        <summary style="cursor: pointer; color: #2271b1;"><?php _e('Ver lista', 'wpp-adi-feed'); ?></summary>
                        <div style="margin-top: 8px; padding: 8px; background: #fff; border-radius: 4px; max-height: 200px; overflow-y: auto;">
                            <code style="font-size: 11px; line-height: 1.6;">
                                <?php echo implode(', ', $valid_tags); ?>
                            </code>
                        </div>
                    </details>
                </span>
            </div>
        <?php endif; ?>

        <div class="status-card status-success">
            <span class="status-label"><?php _e('Limpeza de Logs', 'wpp-adi-feed'); ?></span>
            <?php if ($is_schedule_active): ?>
                <span class="status-value" style="color: #00a32a; font-weight: 600;">✓ <?php _e('Ativo', 'wpp-adi-feed'); ?></span>
                <p style="margin: 8px 0 0 0; font-size: 12px; color: #666;">
                    <?php
                    $next_run = wp_next_scheduled('wpp_adi_feed_daily_log_cleanup');
                    echo sprintf(__('Próxima execução: %s', 'wpp-adi-feed'), date_i18n('d/m/Y H:i', $next_run));
                    ?>
                </p>
                <a href="<?php echo admin_url('admin-post.php?action=toggle_schedule&status=deactivate'); ?>" class="button button-small" style="margin-top: 8px;">
                    <?php _e('Desativar', 'wpp-adi-feed'); ?>
                </a>
            <?php else: ?>
                <span class="status-value" style="color: #dba617; font-weight: 600;">⚠ <?php _e('Inativo', 'wpp-adi-feed'); ?></span>
                <a href="<?php echo admin_url('admin-post.php?action=toggle_schedule&status=activate'); ?>" class="button button-primary button-small" style="margin-top: 8px;">
                    <?php _e('Ativar', 'wpp-adi-feed'); ?>
                </a>
            <?php endif; ?>
        </div>

        <div class="status-card">
            <span class="status-label"><?php _e('URLs Configuradas', 'wpp-adi-feed'); ?></span>
            <div style="margin-top: 8px; font-size: 12px;">
                <p style="margin: 4px 0;">
                    <strong><?php _e('API:', 'wpp-adi-feed'); ?></strong><br>
                    <code style="font-size: 11px;"><?php echo esc_url(wpp_adi_feed_get_api_base_url()); ?></code>
                </p>
                <p style="margin: 4px 0;">
                    <strong><?php _e('CDN Script:', 'wpp-adi-feed'); ?></strong><br>
                    <code style="font-size: 11px;"><?php echo esc_url(wpp_adi_feed_get_cdn_script_url()); ?></code>
                </p>
            </div>
        </div>
<?php
    }

    // Callbacks dos campos
    public function api_section_callback()
    {
        echo '<p>' . __('Configure as URLs e opções de integração com a API.', 'wpp-adi-feed') . '</p>';
    }

    public function api_base_url_callback()
    {
        $value = isset($this->settings['api_base_url']) ? $this->settings['api_base_url'] : 'https://relatorios.prportais.com.br/api';
        echo '<input type="url" name="wpp_adi_feed_settings[api_base_url]" value="' . esc_attr($value) . '" size="100" required>';
        echo '<p class="description">' . __('URL base da API para registro de posts e obtenção de tags válidas', 'wpp-adi-feed') . '</p>';
    }

    public function cdn_script_url_callback()
    {
        $value = isset($this->settings['cdn_script_url']) ? $this->settings['cdn_script_url'] : 'https://relatorio-adi.ddev.site/storage/cdn-adi/news-tracker-wp-min.js';
        echo '<input type="url" name="wpp_adi_feed_settings[cdn_script_url]" value="' . esc_attr($value) . '" size="100" required>';
        echo '<p class="description">' . __('URL do script CDN que será carregado nos posts individuais', 'wpp-adi-feed') . '</p>';
    }

    public function request_timeout_callback()
    {
        $value = isset($this->settings['request_timeout']) ? $this->settings['request_timeout'] : 10;
        echo '<input type="number" name="wpp_adi_feed_settings[request_timeout]" value="' . esc_attr($value) . '" min="5" max="60">';
    }

    public function cache_duration_callback()
    {
        $value = isset($this->settings['cache_duration']) ? $this->settings['cache_duration'] : 3600;
        echo '<input type="number" name="wpp_adi_feed_settings[cache_duration]" value="' . esc_attr($value) . '" min="300" max="86400">';
        echo '<p class="description">' . __('300 = 5 minutos, 3600 = 1 hora, 86400 = 24 horas', 'wpp-adi-feed') . '</p>';
    }

    public function auto_register_callback()
    {
        $value = isset($this->settings['auto_register']) ? $this->settings['auto_register'] : true;
        echo '<input type="checkbox" name="wpp_adi_feed_settings[auto_register]" value="1" ' . checked(1, $value, false) . '>';
        echo '<label>' . __('Registrar posts automaticamente quando salvos', 'wpp-adi-feed') . '</label>';
    }

    public function log_errors_callback()
    {
        $value = isset($this->settings['log_errors']) ? $this->settings['log_errors'] : true;
        echo '<input type="checkbox" name="wpp_adi_feed_settings[log_errors]" value="1" ' . checked(1, $value, false) . '>';
        echo '<label>' . __('Registrar erros no log do WordPress', 'wpp-adi-feed') . '</label>';
    }

    public function log_retention_days_callback()
    {
        $value = isset($this->settings['log_retention_days']) ? intval($this->settings['log_retention_days']) : 7;
        echo '<input type="number" name="wpp_adi_feed_settings[log_retention_days]" value="' . esc_attr($value) . '" min="1" max="365"> ';
        echo '<span class="description">Logs com mais de X dias serão automaticamente removidos. Ex: 7 dias mantém apenas logs da última semana.</span>';
    }

    public function sanitize_settings($input)
    {
        $sanitized = array();

        // Sanitizar log_retention_days como inteiro
        if (isset($input['log_retention_days'])) {
            $sanitized['log_retention_days'] = intval($input['log_retention_days']);
            // Garantir valor mínimo de 1
            if ($sanitized['log_retention_days'] < 1) {
                $sanitized['log_retention_days'] = 1;
            }
        }

        // Sanitizar outros campos
        if (isset($input['api_base_url'])) {
            $sanitized['api_base_url'] = sanitize_url($input['api_base_url']);
        }

        if (isset($input['cdn_script_url'])) {
            $sanitized['cdn_script_url'] = sanitize_url($input['cdn_script_url']);
        }

        if (isset($input['request_timeout'])) {
            $sanitized['request_timeout'] = intval($input['request_timeout']);
        }

        if (isset($input['cache_duration'])) {
            $sanitized['cache_duration'] = intval($input['cache_duration']);
        }

        // Corrigir checkboxes: se não vier no input, define como 0
        $sanitized['auto_register'] = isset($input['auto_register']) ? 1 : 0;
        $sanitized['log_errors'] = isset($input['log_errors']) ? 1 : 0;

        return $sanitized;
    }

    public function schedule_section_callback()
    {
        echo '<p>' . __('Gerencie o agendamento da limpeza de logs.', 'wpp-adi-feed') . '</p>';
    }

    public function toggle_schedule_callback()
    {
        // Este campo agora é gerenciado visualmente no painel de status
        // Não precisa renderizar nada aqui
    }

    /**
     * Testar conexão
     */
    public function test_connection()
    {
        if (!wp_verify_nonce($_POST['wpp_adi_feed_nonce'], 'wpp_adi_feed_test_connection')) {
            wp_die(__('Acesso negado', 'wpp-adi-feed'));
        }

        if (!current_user_can('manage_options')) {
            wp_die(__('Acesso negado', 'wpp-adi-feed'));
        }

        $api_client = $this->get_api_client();
        $success = $api_client ? $api_client->test_connection() : false;
        $message = $success ? 'connection_success' : 'connection_failed';

        wp_redirect(add_query_arg('message', $message, admin_url('options-general.php?page=wpp-adi-feed')));
        exit;
    }

    /**
     * Limpar cache
     */
    public function clear_cache()
    {
        if (!wp_verify_nonce($_POST['wpp_adi_feed_nonce'], 'wpp_adi_feed_clear_cache')) {
            wp_die(__('Acesso negado', 'wpp-adi-feed'));
        }

        if (!current_user_can('manage_options')) {
            wp_die(__('Acesso negado', 'wpp-adi-feed'));
        }

        $api_client = $this->get_api_client();
        if ($api_client) {
            $api_client->clear_tags_cache();
        }

        wp_redirect(add_query_arg('message', 'cache_cleared', admin_url('options-general.php?page=wpp-adi-feed')));
        exit;
    }

    /**
     * Ativar/Desativar Limpeza de Logs
     */
    public function toggle_schedule_handler()
    {
        if (!current_user_can('manage_options')) {
            wp_die(__('Você não tem permissão para realizar esta ação.', 'wpp-adi-feed'));
        }

        $status = isset($_GET['status']) ? sanitize_text_field($_GET['status']) : '';

        if ($status === 'activate') {
            if (!wp_next_scheduled('wpp_adi_feed_daily_log_cleanup')) {
                wp_schedule_event(strtotime('01:00:00'), 'daily', 'wpp_adi_feed_daily_log_cleanup');
            }
        } elseif ($status === 'deactivate') {
            // Remover todos os eventos agendados para esse hook
            wp_clear_scheduled_hook('wpp_adi_feed_daily_log_cleanup');
        }

        wp_redirect(admin_url('options-general.php?page=wpp-adi-feed'));
        exit;
    }

    public function check_schedule_status_notice()
    {
        // Exibir aviso apenas na página de configurações do plugin
        if (!wp_next_scheduled('wpp_adi_feed_daily_log_cleanup') && isset($_GET['page']) && $_GET['page'] === 'wpp-adi-feed') {
            echo '<div class="notice notice-warning is-dismissible"><p><strong>WPP ADI Feed:</strong> O evento de limpeza diária de logs está <span style="color:#dba617">inativo</span>. <a href="' . esc_url(admin_url('admin-post.php?action=toggle_schedule&status=activate')) . '" class="button button-primary">Ativar agora</a></p></div>';
        }
    }
}
