<?php

/**
 * Cliente API para integração com serviços externos
 */

if (!defined('ABSPATH')) {
    exit;
}

class WPP_ADI_Feed_API_Client
{

    private $settings;
    private $logger;

    public function __construct($logger = null)
    {
        $this->settings = get_option('wpp_adi_feed_settings', array());
        $this->logger = $logger;
    }

    /**
     * Obter URL base da API usando função global
     */
    private function get_api_base_url()
    {
        return wpp_adi_feed_get_api_base_url();
    }

    /**
     * Obter tags válidas da API com cache
     */
    public function get_valid_tags()
    {
        // Verificar cache primeiro
        $cached_tags = get_transient('wpp_adi_feed_valid_tags');
        if ($cached_tags !== false) {
            return $cached_tags;
        }

        $api_url = $this->get_api_base_url() . '/tag-list';
        error_log("WPP ADI Feed: Tentando obter tags da URL: {$api_url}");

        $response = wp_remote_get($api_url, array(
            'timeout' => $this->get_timeout(),
            'headers' => array(
                'User-Agent' => 'WPP-ADI-Feed/' . WPP_ADI_FEED_VERSION,
                'Accept' => 'application/json'
            )
        ));

        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            error_log("WPP ADI Feed: ERRO na requisição get_valid_tags: {$error_message}");
            $this->log_api_error('get_valid_tags', $error_message);
            return false;
        }

        $response_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        error_log("WPP ADI Feed: Resposta get_valid_tags - HTTP {$response_code}, Body: " . substr($body, 0, 200));

        if ($response_code !== 200) {
            error_log("WPP ADI Feed: HTTP {$response_code} na requisição get_valid_tags");
            $this->log_api_error('get_valid_tags', "HTTP {$response_code}");
            return false;
        }

        $body = wp_remote_retrieve_body($response);
        $tags = json_decode($body, true);

        // Validação aprimorada do schema da resposta
        if (!$this->validate_tags_schema($tags)) {
            $this->log_api_error('get_valid_tags', 'Formato de resposta inválido ou schema incorreto');
            return false;
        }

        // Salvar no cache
        $cache_duration = $this->get_cache_duration();
        set_transient('wpp_adi_feed_valid_tags', $tags, $cache_duration);

        // Log de sucesso
        if ($this->logger) {
            $this->logger->log_tags_load(true, count($tags));
        }
        return $tags;
    }

    /**
     * Validar schema da resposta de tags
     */
    private function validate_tags_schema($tags)
    {
        // Verificar se é array
        if (!is_array($tags)) {
            return false;
        }

        // Verificar se não está vazio
        if (empty($tags)) {
            return false;
        }

        // Verificar se todos os elementos são strings válidas
        foreach ($tags as $tag) {
            if (!is_string($tag) || empty(trim($tag))) {
                return false;
            }
        }

        return true;
    }

    /**
     * Preparar dados do post para envio
     */
    public function prepare_post_data($post_id, $valid_tags_string = '')
    {
        $post = get_post($post_id);
        if (!$post) {
            return false;
        }

        $domain = parse_url(get_site_url(), PHP_URL_HOST);
        $publish_date = get_the_date('Y-m-d H:i:s', $post_id); // Formato MySQL datetime para Laravel
        $permalink = get_permalink($post_id);
        $title = get_the_title($post_id);

        // Validar dados conforme esperado pelo Laravel
        $data = array(
            'title' => $title,
            'permalink' => $permalink,
            'tag' => $valid_tags_string,
            'domain' => $domain,
            'datetime' => $publish_date
        );

        // Sanitizar e validar conforme regras do Laravel
        // domain: required|string
        if (empty($data['domain']) || !is_string($data['domain'])) {
            error_log("WPP ADI Feed: Domain inválido: " . $data['domain']);
            return false;
        }

        // title: required|string|max:255
        if (empty($data['title']) || !is_string($data['title']) || strlen($data['title']) > 255) {
            error_log("WPP ADI Feed: Title inválido - vazio ou muito longo: " . substr($data['title'], 0, 50) . "...");
            return false;
        }

        // permalink: required|string
        if (empty($data['permalink']) || !is_string($data['permalink'])) {
            error_log("WPP ADI Feed: Permalink inválido: " . $data['permalink']);
            return false;
        }

        // datetime: required|date (formato Y-m-d H:i:s)
        if (empty($data['datetime']) || !strtotime($data['datetime'])) {
            error_log("WPP ADI Feed: Datetime inválido: " . $data['datetime']);
            return false;
        }

        // tag: nullable|string (opcional)
        if (!empty($data['tag']) && !is_string($data['tag'])) {
            error_log("WPP ADI Feed: Tag deve ser string: " . $data['tag']);
            return false;
        }

        // error_log("WPP ADI Feed: Dados preparados para API - Title: {$data['title']},Tag: {$data['tag']},Permalink: {$data['permalink']}, Domain: {$data['domain']}, Datetime: {$data['datetime']}");

        return $data;
    }

    /**
     * Registrar post na API
     */
    public function register_post($post_data)
    {
        error_log("WPP ADI Feed: Iniciando registro do post na API - Título: " . ($post_data['title'] ?? 'N/A'));

        $api_url = $this->get_api_base_url() . '/posts/wordpress';
        // error_log("WPP ADI Feed: URL da API para registro: {$api_url}");

        $response = wp_remote_post($api_url, array(
            'timeout' => $this->get_timeout(),
            'headers' => array(
                'Content-Type' => 'application/json',
                'User-Agent' => 'WPP-ADI-Feed/' . WPP_ADI_FEED_VERSION,
                'Accept' => 'application/json'
            ),
            'body' => wp_json_encode($post_data)
        ));

        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            // error_log("WPP ADI Feed: ERRO ao enviar post para API - {$error_message}");

            $this->log_api_error('register_post', $error_message);
            if ($this->logger) {
                $this->logger->log_api_request($api_url, false, null, $error_message);
            }

            return false;
        }

        $response_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $response_data = json_decode($body, true);

        // error_log("WPP ADI Feed: Resposta da API - HTTP {$response_code}, Body length: " . strlen($body));

        // Validar resposta JSON
        if (json_last_error() !== JSON_ERROR_NONE) {
            // error_log("WPP ADI Feed: ERRO - Resposta JSON inválida da API: " . json_last_error_msg());
            // error_log("WPP ADI Feed: Corpo da resposta (primeiros 500 chars): " . substr($body, 0, 500));
            $this->log_api_error('register_post', 'Resposta JSON inválida: ' . json_last_error_msg());
            if ($this->logger) {
                $this->logger->log_api_request($api_url, false, null, 'JSON inválido: ' . substr($body, 0, 200));
            }
            return false;
        }

        $success = in_array($response_code, array(200, 201));

        // if ($success) {
        //     error_log("WPP ADI Feed: SUCESSO - Post enviado para API com código HTTP {$response_code}");
        // } else {
        //     error_log("WPP ADI Feed: FALHA - Post NÃO enviado para API - HTTP {$response_code}: {$body}");
        // }

        if ($this->logger) {
            $this->logger->log_api_request(
                $api_url,
                $success,
                $response_data,
                !$success ? "HTTP {$response_code}: {$body}" : null
            );
        }

        if (!$success) {
            $this->log_api_error('register_post', "HTTP {$response_code}: {$body}");
        }

        return $success;
    }

    /**
     * Testar conexão com a API
     */
    public function test_connection()
    {
        $api_url = $this->get_api_base_url() . '/status';

        $response = wp_remote_get($api_url, array(
            'timeout' => $this->get_timeout(),
            'headers' => array(
                'User-Agent' => 'WPP-ADI-Feed/' . WPP_ADI_FEED_VERSION,
                'Accept' => 'application/json'
            )
        ));

        if (is_wp_error($response)) {
            $this->log_api_error('test_connection', $response->get_error_message());
            return false;
        }

        $response_code = wp_remote_retrieve_response_code($response);
        $success = $response_code === 200;

        if ($this->logger) {
            $body = wp_remote_retrieve_body($response);
            $this->logger->log_api_request(
                $api_url,
                $success,
                json_decode($body, true),
                !$success ? "HTTP {$response_code}" : null
            );
        }

        return $success;
    }

    /**
     * Limpar cache de tags
     */
    public function clear_tags_cache()
    {
        return delete_transient('wpp_adi_feed_valid_tags');
    }

    /**
     * Obter timeout das requisições
     */
    private function get_timeout()
    {
        return isset($this->settings['request_timeout']) ?
            intval($this->settings['request_timeout']) : 10;
    }

    /**
     * Obter duração do cache
     */
    private function get_cache_duration()
    {
        return isset($this->settings['cache_duration']) ?
            intval($this->settings['cache_duration']) : 3600;
    }

    /**
     * Log de erros da API (compatibilidade)
     */
    private function log_api_error($action, $message)
    {
        if (isset($this->settings['log_errors']) && $this->settings['log_errors']) {
            error_log("WPP ADI Feed API [{$action}]: {$message}");
        }
    }

    /**
     * Obter estatísticas de uso da API (novo)
     */
    public function get_api_stats()
    {
        $stats = array(
            'api_base_url' => $this->get_api_base_url(),
            'cdn_script_url' => wpp_adi_feed_get_cdn_script_url(),
            'cache_status' => get_transient('wpp_adi_feed_valid_tags') !== false,
            'last_cache_update' => get_option('_transient_timeout_wpp_adi_feed_valid_tags'),
            'timeout_setting' => $this->get_timeout(),
            'cache_duration' => $this->get_cache_duration()
        );

        return $stats;
    }

    /**
     * Validar configurações da API
     */
    public function validate_settings()
    {
        $errors = array();

        $api_url = $this->get_api_base_url();
        if (empty($api_url) || !filter_var($api_url, FILTER_VALIDATE_URL)) {
            $errors[] = 'URL da API inválida';
        }

        $cdn_url = wpp_adi_feed_get_cdn_script_url();
        if (empty($cdn_url) || !filter_var($cdn_url, FILTER_VALIDATE_URL)) {
            $errors[] = 'URL do script CDN inválida';
        }

        if ($this->get_timeout() < 5 || $this->get_timeout() > 60) {
            $errors[] = 'Timeout deve estar entre 5 e 60 segundos';
        }

        if ($this->get_cache_duration() < 300) {
            $errors[] = 'Duração do cache deve ser de pelo menos 300 segundos (5 minutos)';
        }

        return $errors;
    }
}
