<?php

/**
 * Página Administrativa Simplificada de Logs
 * Foco em: posts enviados, tags válidas, título, slug, data
 */

if (!defined('ABSPATH')) {
    exit;
}

class WPP_ADI_Feed_Log_Admin
{

    private $logger;

    public function __construct($logger)
    {
        $this->logger = $logger;

        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_post_wpp_adi_feed_clear_logs', array($this, 'clear_logs'));
        add_action('admin_post_wpp_adi_feed_export_logs', array($this, 'export_logs'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_styles'));
    }

    /**
     * Adicionar menu de logs
     */
    public function add_admin_menu()
    {
        add_submenu_page(
            'options-general.php',
            __('ADI Feed - Logs', 'wpp-adi-feed'),
            __('ADI Feed Logs', 'wpp-adi-feed'),
            'manage_options',
            'wpp-adi-feed-logs',
            array($this, 'logs_page')
        );
    }

    /**
     * Enfileirar estilos
     */
    public function enqueue_admin_styles($hook)
    {
        if ($hook !== 'settings_page_wpp-adi-feed-logs') {
            return;
        }

        echo '<style>
        .wpp-adi-stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin: 20px 0;
        }
        .wpp-adi-stat-card {
            background: #fff;
            border: 1px solid #ddd;
            border-radius: 6px;
            padding: 20px;
            text-align: center;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
        }
        .wpp-adi-stat-card h3 {
            margin: 0 0 10px 0;
            font-size: 14px;
            color: #666;
        }
        .wpp-adi-stat-number {
            font-size: 32px;
            font-weight: bold;
            color: #333;
        }
        .wpp-adi-stat-card.success .wpp-adi-stat-number { color: #46b450; }
        .wpp-adi-stat-card.errorr .wpp-adi-stat-number { color: #dc3232; }
        .wpp-adi-stat-card.warning .wpp-adi-stat-number { color: #ffb900; }
        
        .wpp-adi-log-filters {
            background: #f9f9f9;
            border: 1px solid #ddd;
            border-radius: 6px;
            padding: 15px;
            margin: 20px 0;
        }
        .filter-row {
            display: flex;
            align-items: center;
            gap: 15px;
            flex-wrap: wrap;
        }
        .filter-row label {
            font-weight: 600;
            margin: 0;
        }
        .filter-row input, .filter-row select {
            margin: 0;
        }
        
        .wpp-adi-logs-table .column-tags {
            width: 200px;
        }
        .wpp-adi-logs-table .column-date {
            width: 140px;
        }
        .wpp-adi-logs-table .column-status {
            width: 80px;
        }
        
        .log-status-success { 
            color: #46b450; 
            font-weight: bold;
        }
        .log-status-error { 
            color: #dc3232; 
            font-weight: bold;
        }
        .log-status-warning { 
            color: #ffb900; 
            font-weight: bold;
        }
        .log-status-info { 
            color: #0073aa; 
            font-weight: bold;
        }
        
        .log-tags {
            display: flex;
            flex-wrap: wrap;
            gap: 4px;
        }
        .log-tag {
            background: #e1f5fe;
            color: #0277bd;
            padding: 2px 6px;
            border-radius: 3px;
            font-size: 11px;
        }
        
        .recent-posts-section {
            background: #fff;
            border: 1px solid #ddd;
            border-radius: 6px;
            padding: 20px;
            margin: 20px 0;
        }
        </style>';
    }

    /**
     * Página de logs
     */
    public function logs_page()
    {
        $current_action = isset($_GET['action_filter']) ? sanitize_text_field($_GET['action_filter']) : '';
        $current_level = isset($_GET['level']) ? sanitize_text_field($_GET['level']) : '';
        $current_search = isset($_GET['search']) ? sanitize_text_field($_GET['search']) : '';
        $current_date_from = isset($_GET['date_from']) ? sanitize_text_field($_GET['date_from']) : '';
        $current_date_to = isset($_GET['date_to']) ? sanitize_text_field($_GET['date_to']) : '';
        $current_page = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1;

        $per_page = 25;
        $offset = ($current_page - 1) * $per_page;

        $filter_args = array(
            'action' => $current_action,
            'level' => $current_level,
            'search' => $current_search,
            'date_from' => $current_date_from,
            'date_to' => $current_date_to,
            'limit' => $per_page,
            'offset' => $offset
        );

        $logs = $this->logger->get_logs($filter_args);
        $total_logs = $this->logger->count_logs($filter_args);
        $stats = $this->logger->get_log_stats();
        $recent_posts = $this->logger->get_recent_sent_posts(5);

?>
        <div class="wrap">
            <h1>
                <?php _e('ADI Feed - Logs de Envio', 'wpp-adi-feed'); ?>
                <a href="<?php echo wp_nonce_url(admin_url('admin-post.php?action=wpp_adi_feed_export_logs'), 'wpp_adi_feed_export_logs'); ?>"
                    class="page-title-action">
                    <?php _e('Exportar CSV', 'wpp-adi-feed'); ?>
                </a>
            </h1>

            <?php $this->show_notices(); ?>

            <!-- Estatísticas Principais -->
            <div class="wpp-adi-stats-grid">
                <div class="wpp-adi-stat-card success">
                    <h3><?php _e('Posts Enviados', 'wpp-adi-feed'); ?></h3>
                    <div class="wpp-adi-stat-number"><?php echo number_format($stats['posts_sent']); ?></div>
                </div>

                <div class="wpp-adi-stat-card errorr">
                    <h3><?php _e('Posts Rejeitados', 'wpp-adi-feed'); ?></h3>
                    <div class="wpp-adi-stat-number"><?php echo number_format($stats['posts_rejected']); ?></div>
                </div>
                <div class="wpp-adi-stat-card warning">
                    <h3><?php _e('Avisos', 'wpp-adi-feed'); ?></h3>
                    <div class="wpp-adi-stat-number"><?php echo number_format($stats['by_level']['warning']); ?></div>
                </div>
                <div class="wpp-adi-stat-card">
                    <h3><?php _e('Total de Logs', 'wpp-adi-feed'); ?></h3>
                    <div class="wpp-adi-stat-number"><?php echo number_format($stats['total']); ?></div>
                </div>
            </div>

            <!-- Últimos Posts Enviados -->
            <?php if (!empty($recent_posts)): ?>
                <div class="recent-posts-section">
                    <h3><?php _e('Últimos Posts Enviados com Sucesso', 'wpp-adi-feed'); ?></h3>
                    <table class="wp-list-table widefat">
                        <thead>
                            <tr>
                                <th><?php _e('Título', 'wpp-adi-feed'); ?></th>
                                <th><?php _e('Tags Válidas', 'wpp-adi-feed'); ?></th>
                                <th><?php _e('Data Publicação', 'wpp-adi-feed'); ?></th>
                                <th><?php _e('Enviado em', 'wpp-adi-feed'); ?></th>
                                <th><?php _e('Link', 'wpp-adi-feed'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($recent_posts as $post): ?>
                                <tr>
                                    <td><strong><?php echo esc_html($post->post_title); ?></strong></td>
                                    <td>
                                        <?php if (!empty($post->valid_tags)): ?>
                                            <div class="log-tags">
                                                <?php foreach ($post->valid_tags as $tag): ?>
                                                    <span class="log-tag"><?php echo esc_html($tag); ?></span>
                                                <?php endforeach; ?>
                                            </div>
                                        <?php else: ?>
                                            <em><?php _e('Nenhuma tag', 'wpp-adi-feed'); ?></em>
                                        <?php endif; ?>
                                    </td>
                                    <td><?php echo esc_html(mysql2date('d/m/Y H:i', $post->post_date)); ?></td>
                                    <td><?php echo esc_html(mysql2date('d/m/Y H:i', $post->timestamp)); ?></td>
                                    <td>
                                        <?php if (isset($post->post_url)): ?>
                                            <a href="<?php echo esc_url($post->post_url); ?>" target="_blank">Ver</a>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>

            <!-- Filtros -->
            <div class="wpp-adi-log-filters">
                <form method="get" action="">
                    <input type="hidden" name="page" value="wpp-adi-feed-logs">

                    <div class="filter-row">
                        <label><?php _e('Ação:', 'wpp-adi-feed'); ?></label>
                        <select name="action_filter">
                            <option value=""><?php _e('Todas', 'wpp-adi-feed'); ?></option>
                            <option value="post_registration" <?php selected($current_action, 'post_registration'); ?>>
                                <?php _e('Registro de Posts', 'wpp-adi-feed'); ?>
                            </option>
                            <option value="tag_validation" <?php selected($current_action, 'tag_validation'); ?>>
                                <?php _e('Validação de Tags', 'wpp-adi-feed'); ?>
                            </option>
                            <option value="api_request" <?php selected($current_action, 'api_request'); ?>>
                                <?php _e('Requisições API', 'wpp-adi-feed'); ?>
                            </option>
                        </select>

                        <label><?php _e('Status:', 'wpp-adi-feed'); ?></label>
                        <select name="level">
                            <option value=""><?php _e('Todos', 'wpp-adi-feed'); ?></option>
                            <option value="success" <?php selected($current_level, 'success'); ?>>
                                <?php _e('Sucesso', 'wpp-adi-feed'); ?>
                            </option>
                            <option value="error" <?php selected($current_level, 'error'); ?>>
                                <?php _e('Erro', 'wpp-adi-feed'); ?>
                            </option>
                            <option value="warning" <?php selected($current_level, 'warning'); ?>>
                                <?php _e('Aviso', 'wpp-adi-feed'); ?>
                            </option>
                            <option value="info" <?php selected($current_level, 'info'); ?>>
                                <?php _e('Info', 'wpp-adi-feed'); ?>
                            </option>
                        </select>

                        <label><?php _e('Buscar:', 'wpp-adi-feed'); ?></label>
                        <input type="text" name="search" value="<?php echo esc_attr($current_search); ?>"
                            placeholder="<?php _e('Título do post...', 'wpp-adi-feed'); ?>">

                        <label><?php _e('De:', 'wpp-adi-feed'); ?></label>
                        <input type="date" name="date_from" value="<?php echo esc_attr($current_date_from); ?>">

                        <label><?php _e('Até:', 'wpp-adi-feed'); ?></label>
                        <input type="date" name="date_to" value="<?php echo esc_attr($current_date_to); ?>">

                        <input type="submit" class="button" value="<?php _e('Filtrar', 'wpp-adi-feed'); ?>">
                        <a href="<?php echo admin_url('options-general.php?page=wpp-adi-feed-logs'); ?>"
                            class="button"><?php _e('Limpar', 'wpp-adi-feed'); ?></a>
                    </div>
                </form>
            </div>

            <!-- Ações rápidas -->
            <p>
            <form method="post" action="<?php echo admin_url('admin-post.php'); ?>" style="display: inline;">
                <?php wp_nonce_field('wpp_adi_feed_clear_logs', 'wpp_adi_feed_nonce'); ?>
                <input type="hidden" name="action" value="wpp_adi_feed_clear_logs">
                <input type="submit" class="button button-secondary"
                    value="<?php _e('Limpar Todos os Logs', 'wpp-adi-feed'); ?>"
                    onclick="return confirm('<?php _e('Tem certeza que deseja limpar todos os logs?', 'wpp-adi-feed'); ?>')">
            </form>
            </p>

            <!-- Tabela de logs -->
            <?php if (empty($logs)): ?>
                <div class="notice notice-info">
                    <p><?php _e('Nenhum log encontrado com os filtros selecionados.', 'wpp-adi-feed'); ?></p>
                </div>
            <?php else: ?>
                <table class="wp-list-table widefat fixed striped wpp-adi-logs-table">
                    <thead>
                        <tr>
                            <th style="width: 140px;"><?php _e('Data/Hora', 'wpp-adi-feed'); ?></th>
                            <th style="width: 80px;"><?php _e('Status', 'wpp-adi-feed'); ?></th>
                            <th style="width: 120px;"><?php _e('Ação', 'wpp-adi-feed'); ?></th>
                            <th><?php _e('Título do Post', 'wpp-adi-feed'); ?></th>
                            <th style="width: 200px;"><?php _e('Tags Válidas', 'wpp-adi-feed'); ?></th>
                            <th style="width: 140px;"><?php _e('Data Pub.', 'wpp-adi-feed'); ?></th>
                            <th style="width: 80px;"><?php _e('Link', 'wpp-adi-feed'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($logs as $log): ?>
                            <tr>
                                <td>
                                    <small><?php echo esc_html(mysql2date('d/m/Y H:i', $log->timestamp)); ?></small>
                                </td>
                                <td>
                                    <span class="log-status-<?php echo esc_attr($log->level); ?>">
                                        <?php echo $this->get_level_label($log->level); ?>
                                    </span>
                                </td>
                                <td>
                                    <small><?php echo $this->get_action_label($log->action); ?></small>
                                </td>
                                <td>
                                    <?php if ($log->post_title): ?>
                                        <strong><?php echo esc_html($log->post_title); ?></strong>
                                        <?php if ($log->post_id): ?>
                                            <br><small>ID: <?php echo $log->post_id; ?></small>
                                        <?php endif; ?>
                                    <?php else: ?>
                                        <em><?php echo esc_html($log->message); ?></em>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php if ($log->valid_tags): ?>
                                        <?php
                                        $tags = json_decode($log->valid_tags, true);
                                        if (is_array($tags) && !empty($tags)):
                                        ?>
                                            <div class="log-tags">
                                                <?php foreach ($tags as $tag): ?>
                                                    <span class="log-tag"><?php echo esc_html($tag); ?></span>
                                                <?php endforeach; ?>
                                            </div>
                                        <?php endif; ?>
                                    <?php else: ?>
                                        <small class="description">-</small>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php if ($log->post_date): ?>
                                        <small><?php echo esc_html(mysql2date('d/m/Y H:i', $log->post_date)); ?></small>
                                    <?php else: ?>
                                        <small class="description">-</small>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php if ($log->post_id && $log->post_slug): ?>
                                        <?php $post_url = get_permalink($log->post_id); ?>
                                        <?php if ($post_url): ?>
                                            <a href="<?php echo esc_url($post_url); ?>" target="_blank" title="<?php echo esc_attr($log->post_slug); ?>">
                                                Ver
                                            </a>
                                        <?php else: ?>
                                            <small><?php echo esc_html($log->post_slug); ?></small>
                                        <?php endif; ?>
                                    <?php else: ?>
                                        <small class="description">-</small>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>

                <!-- Paginação -->
                <?php if ($total_logs > $per_page): ?>
                    <div class="tablenav bottom">
                        <div class="tablenav-pages">
                            <span class="displaying-num">
                                <?php printf(_n('%s item', '%s itens', $total_logs, 'wpp-adi-feed'), number_format_i18n($total_logs)); ?>
                            </span>
                            <?php
                            $total_pages = ceil($total_logs / $per_page);

                            $pagination_args = array(
                                'base' => add_query_arg('paged', '%#%'),
                                'format' => '',
                                'prev_text' => '&laquo; ' . __('Anterior', 'wpp-adi-feed'),
                                'next_text' => __('Próxima', 'wpp-adi-feed') . ' &raquo;',
                                'total' => $total_pages,
                                'current' => $current_page,
                                'show_all' => false,
                                'end_size' => 1,
                                'mid_size' => 2
                            );

                            echo paginate_links($pagination_args);
                            ?>
                        </div>
                    </div>
                <?php endif; ?>
            <?php endif; ?>
        </div>
<?php
    }

    /**
     * Mostrar notices
     */
    private function show_notices()
    {
        if (isset($_GET['message'])) {
            $message = sanitize_text_field($_GET['message']);
            switch ($message) {
                case 'logs_cleared':
                    echo '<div class="notice notice-success is-dismissible">
                            <p>' . __('Todos os logs foram limpos com sucesso!', 'wpp-adi-feed') . '</p>
                          </div>';
                    break;
                case 'export_error':
                    echo '<div class="notice notice-error is-dismissible">
                            <p>' . __('Erro ao exportar logs.', 'wpp-adi-feed') . '</p>
                          </div>';
                    break;
            }
        }
    }

    /**
     * Obter rótulo do nível
     */
    private function get_level_label($level)
    {
        $labels = array(
            'success' => __('✓ Sucesso', 'wpp-adi-feed'),
            'error' => __('✗ Erro', 'wpp-adi-feed'),
            'warning' => __('⚠ Aviso', 'wpp-adi-feed'),
            'info' => __('ℹ Info', 'wpp-adi-feed')
        );

        return isset($labels[$level]) ? $labels[$level] : $level;
    }

    /**
     * Obter rótulo da ação
     */
    private function get_action_label($action)
    {
        $labels = array(
            'post_registration' => __('Registro', 'wpp-adi-feed'),
            'tag_validation' => __('Validação', 'wpp-adi-feed'),
            'api_request' => __('API', 'wpp-adi-feed'),
            'tags_load' => __('Tags', 'wpp-adi-feed')
        );

        return isset($labels[$action]) ? $labels[$action] : $action;
    }

    /**
     * Limpar logs
     */
    public function clear_logs()
    {
        if (!wp_verify_nonce($_POST['wpp_adi_feed_nonce'], 'wpp_adi_feed_clear_logs')) {
            wp_die(__('Acesso negado', 'wpp-adi-feed'));
        }

        if (!current_user_can('manage_options')) {
            wp_die(__('Acesso negado', 'wpp-adi-feed'));
        }

        $this->logger->clear_all_logs();

        wp_redirect(add_query_arg('message', 'logs_cleared', admin_url('options-general.php?page=wpp-adi-feed-logs')));
        exit;
    }

    /**
     * Exportar logs
     */
    public function export_logs()
    {
        if (!wp_verify_nonce($_GET['_wpnonce'], 'wpp_adi_feed_export_logs')) {
            wp_die(__('Acesso negado', 'wpp-adi-feed'));
        }

        if (!current_user_can('manage_options')) {
            wp_die(__('Acesso negado', 'wpp-adi-feed'));
        }

        try {
            $this->logger->export_logs_csv();
        } catch (Exception $e) {
            wp_redirect(add_query_arg('message', 'export_error', admin_url('options-general.php?page=wpp-adi-feed-logs')));
            exit;
        }
    }
}
