<?php

/**
 * Sistema de Log Simplificado para WPP ADI Feed
 * Foco em: tags/categorias, título, slug, data de publicação
 */

if (!defined('ABSPATH')) {
    exit;
}

class WPP_ADI_Feed_Logger
{

    private $log_table_name;
    private $settings;

    const LOG_LEVEL_INFO = 'info';
    const LOG_LEVEL_WARNING = 'warning';
    const LOG_LEVEL_ERROR = 'error';
    const LOG_LEVEL_SUCCESS = 'success';

    public function __construct()
    {
        global $wpdb;
        $this->log_table_name = $wpdb->prefix . 'wpp_adi_feed_logs';
        $this->settings = get_option('wpp_adi_feed_settings', array());

        $this->maybe_create_log_table();
    }

    /**
     * Criar tabela de logs otimizada
     */
    private function maybe_create_log_table()
    {
        global $wpdb;

        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE {$this->log_table_name} (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            timestamp datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            level varchar(20) NOT NULL,
            action varchar(50) NOT NULL,
            message text NOT NULL,
            post_id bigint(20) DEFAULT NULL,
            post_title varchar(255),
            post_slug varchar(200),
            post_date datetime,
            valid_tags text,
            api_response text,
            PRIMARY KEY (id),
            KEY level_action (level, action),
            KEY post_id (post_id),
            KEY timestamp (timestamp)
        ) $charset_collate;";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }

    /**
     * Log específico para registro de post na API
     */
    public function log_post_registration($post_id, $success, $valid_tags = array(), $api_response = null)
    {
        $post = get_post($post_id);
        if (!$post) return false;

        $level = $success ? self::LOG_LEVEL_SUCCESS : self::LOG_LEVEL_ERROR;
        $action = 'post_registration';
        $message = $success
            ? sprintf('Post "%s" enviado com sucesso para a API', $post->post_title)
            : sprintf('Falha ao enviar post "%s" para a API', $post->post_title);

        return $this->insert_log(
            $level,
            $action,
            $message,
            $post_id,
            $post->post_title,
            $post->post_name,
            $post->post_date,
            $valid_tags,
            $api_response
        );
    }

    /**
     * Log para validação de tags
     */
    public function log_tag_validation($post_id, $valid_tags_found, $will_register = false)
    {
        $post = get_post($post_id);
        if (!$post) return false;

        $level = $will_register ? self::LOG_LEVEL_INFO : self::LOG_LEVEL_WARNING;
        $action = 'tag_validation';

        if ($will_register) {
            $message = sprintf(
                'Post "%s" possui tags válidas: %s',
                $post->post_title,
                implode(', ', $valid_tags_found)
            );
        } else {
            $message = sprintf(
                'Post "%s" não possui tags válidas - não será registrado',
                $post->post_title
            );
        }

        return $this->insert_log(
            $level,
            $action,
            $message,
            $post_id,
            $post->post_title,
            $post->post_name,
            $post->post_date,
            $valid_tags_found
        );
    }

    /**
     * Log para requisições da API
     */
    public function log_api_request($endpoint, $success, $response_data = null, $error_message = null)
    {
        $level = $success ? self::LOG_LEVEL_SUCCESS : self::LOG_LEVEL_ERROR;
        $action = 'api_request';

        if ($success) {
            $message = "Requisição API bem-sucedida";
            $api_response = is_array($response_data) ? wp_json_encode($response_data) : $response_data;
        } else {
            $message = "Falha na requisição API: {$endpoint}" . ($error_message ? " - {$error_message}" : '');
            $api_response = $error_message;
        }

        return $this->insert_log($level, $action, $message, null, null, null, null, array(), $api_response);
    }

    /**
     * Log para carregamento de tags válidas
     */
    public function log_tags_load($success, $tags_count = 0, $error_message = null)
    {
        $level = $success ? self::LOG_LEVEL_SUCCESS : self::LOG_LEVEL_ERROR;
        $action = 'tags_load';

        if ($success) {
            $message = "Tags válidas carregadas com sucesso: {$tags_count} tags disponíveis";
        } else {
            $message = "Falha ao carregar tags válidas" . ($error_message ? " - {$error_message}" : '');
        }

        return $this->insert_log($level, $action, $message, null, null, null, null, array(), $error_message);
    }

    /**
     * Inserir log na tabela
     */
    private function insert_log(
        $level,
        $action,
        $message,
        $post_id = null,
        $post_title = null,
        $post_slug = null,
        $post_date = null,
        $valid_tags = array(),
        $api_response = null
    ) {

        if (!$this->is_logging_enabled()) {
            return false;
        }

        global $wpdb;

        $valid_tags_json = !empty($valid_tags) ? wp_json_encode($valid_tags, JSON_UNESCAPED_UNICODE) : null;
        $api_response_text = is_array($api_response) ? wp_json_encode($api_response, JSON_UNESCAPED_UNICODE) : $api_response;

        $result = $wpdb->insert(
            $this->log_table_name,
            array(
                'level' => $level,
                'action' => $action,
                'message' => $message,
                'post_id' => $post_id,
                'post_title' => $post_title,
                'post_slug' => $post_slug,
                'post_date' => $post_date,
                'valid_tags' => $valid_tags_json,
                'api_response' => $api_response_text
            ),
            array('%s', '%s', '%s', '%d', '%s', '%s', '%s', '%s', '%s')
        );

        // Log crítico removido

        // Limpeza automática ocasional
        if (rand(1, 100) <= 2) { // 2% de chance
            $this->cleanup_old_logs();
        }

        return $result !== false;
    }

    /**
     * Obter logs com filtros
     */
    public function get_logs($args = array())
    {
        global $wpdb;

        $defaults = array(
            'level' => '',
            'action' => '',
            'post_id' => 0,
            'limit' => 50,
            'offset' => 0,
            'date_from' => '',
            'date_to' => '',
            'search' => ''
        );

        $args = wp_parse_args($args, $defaults);

        $where_conditions = array('1=1');
        $where_values = array();

        if (!empty($args['level'])) {
            $where_conditions[] = 'level = %s';
            $where_values[] = $args['level'];
        }

        if (!empty($args['action'])) {
            $where_conditions[] = 'action = %s';
            $where_values[] = $args['action'];
        }

        if (!empty($args['post_id'])) {
            $where_conditions[] = 'post_id = %d';
            $where_values[] = $args['post_id'];
        }

        if (!empty($args['date_from'])) {
            $where_conditions[] = 'timestamp >= %s';
            $where_values[] = $args['date_from'] . ' 00:00:00';
        }

        if (!empty($args['date_to'])) {
            $where_conditions[] = 'timestamp <= %s';
            $where_values[] = $args['date_to'] . ' 23:59:59';
        }

        if (!empty($args['search'])) {
            $where_conditions[] = '(message LIKE %s OR post_title LIKE %s)';
            $search_term = '%' . $wpdb->esc_like($args['search']) . '%';
            $where_values[] = $search_term;
            $where_values[] = $search_term;
        }

        $where_clause = implode(' AND ', $where_conditions);

        $sql = "SELECT * FROM {$this->log_table_name} 
                WHERE {$where_clause} 
                ORDER BY timestamp DESC 
                LIMIT %d OFFSET %d";

        $where_values[] = $args['limit'];
        $where_values[] = $args['offset'];

        if (!empty($where_values)) {
            $sql = $wpdb->prepare($sql, $where_values);
        }

        return $wpdb->get_results($sql);
    }

    /**
     * Contar logs
     */
    public function count_logs($args = array())
    {
        global $wpdb;

        $where_conditions = array('1=1');
        $where_values = array();

        if (!empty($args['level'])) {
            $where_conditions[] = 'level = %s';
            $where_values[] = $args['level'];
        }

        if (!empty($args['action'])) {
            $where_conditions[] = 'action = %s';
            $where_values[] = $args['action'];
        }

        if (!empty($args['search'])) {
            $where_conditions[] = '(message LIKE %s OR post_title LIKE %s)';
            $search_term = '%' . $wpdb->esc_like($args['search']) . '%';
            $where_values[] = $search_term;
            $where_values[] = $search_term;
        }

        $where_clause = implode(' AND ', $where_conditions);
        $sql = "SELECT COUNT(*) FROM {$this->log_table_name} WHERE {$where_clause}";

        if (!empty($where_values)) {
            $sql = $wpdb->prepare($sql, $where_values);
        }

        return (int) $wpdb->get_var($sql);
    }

    /**
     * Estatísticas dos logs
     */
    public function get_log_stats($days = 30)
    {
        global $wpdb;

        $date_from = date('Y-m-d', strtotime("-{$days} days"));

        // Stats por nível e ação
        $sql = $wpdb->prepare("
            SELECT 
                level,
                action,
                COUNT(*) as count
            FROM {$this->log_table_name} 
            WHERE timestamp >= %s
            GROUP BY level, action
        ", $date_from . ' 00:00:00');

        $results = $wpdb->get_results($sql);

        $stats = array(
            'total' => 0,
            'by_level' => array(
                'success' => 0,
                'info' => 0,
                'warning' => 0,
                'error' => 0
            ),
            'by_action' => array(
                'post_registration' => 0,
                'tag_validation' => 0,
                'api_request' => 0,
                'tags_load' => 0
            ),
            'posts_sent' => 0,
            'posts_rejected' => 0
        );

        foreach ($results as $row) {
            $stats['total'] += $row->count;
            if (!isset($stats['by_level'][$row->level])) {
                $stats['by_level'][$row->level] = 0;
            }
            $stats['by_level'][$row->level] += $row->count;
            if (!isset($stats['by_action'][$row->action])) {
                $stats['by_action'][$row->action] = 0;
            }
            $stats['by_action'][$row->action] += $row->count;

            // Contar posts enviados vs rejeitados
            if ($row->action === 'post_registration') {
                if ($row->level === 'success') {
                    $stats['posts_sent'] += $row->count;
                } else {
                    $stats['posts_rejected'] += $row->count;
                }
            }
        }

        return $stats;
    }

    /**
     * Obter últimos posts enviados
     */
    public function get_recent_sent_posts($limit = 10)
    {
        global $wpdb;

        $sql = $wpdb->prepare("
            SELECT post_id, post_title, post_slug, post_date, valid_tags, timestamp
            FROM {$this->log_table_name} 
            WHERE action = 'post_registration' 
            AND level = 'success'
            ORDER BY timestamp DESC 
            LIMIT %d
        ", $limit);

        $results = $wpdb->get_results($sql);

        foreach ($results as &$result) {
            if ($result->valid_tags) {
                $result->valid_tags = json_decode($result->valid_tags, true);
            }
            if ($result->post_id) {
                $result->post_url = get_permalink($result->post_id);
            }
        }

        return $results;
    }

    /**
     * Limpar logs antigos
     */
    public function cleanup_old_logs($days = 7)
    {
        global $wpdb;

        $date_limit = date('Y-m-d H:i:s', strtotime("-{$days} days"));

        // Debug: verificar query
        error_log("WPP ADI Feed: Date limit: $date_limit, Days: $days");

        $query = $wpdb->prepare(
            "DELETE FROM {$this->log_table_name} WHERE timestamp < %s",
            $date_limit
        );

        error_log("WPP ADI Feed: Query: $query");

        $result = $wpdb->query($query);

        error_log("WPP ADI Feed: Deleted $result rows");

        return $result;
    }

    /**
     * Limpar todos os logs
     */
    public function clear_all_logs()
    {
        global $wpdb;
        return $wpdb->query("TRUNCATE TABLE {$this->log_table_name}");
    }

    /**
     * Exportar logs essenciais para CSV
     */
    public function export_logs_csv($args = array())
    {
        $logs = $this->get_logs(array_merge($args, array('limit' => 5000)));

        $filename = 'wpp-adi-feed-logs-' . date('Y-m-d-H-i-s') . '.csv';

        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="' . $filename . '"');

        $output = fopen('php://output', 'w');

        // BOM para UTF-8
        fprintf($output, chr(0xEF) . chr(0xBB) . chr(0xBF));

        // Cabeçalhos CSV
        fputcsv($output, array(
            'Data/Hora',
            'Ação',
            'Status',
            'Título do Post',
            'Slug',
            'Data de Publicação',
            'Tags Válidas',
            'Mensagem'
        ), ';');

        foreach ($logs as $log) {
            $valid_tags = '';
            if ($log->valid_tags) {
                $tags_array = json_decode($log->valid_tags, true);
                $valid_tags = is_array($tags_array) ? implode(', ', $tags_array) : $log->valid_tags;
            }

            $status_map = array(
                'success' => 'Sucesso',
                'error' => 'Erro',
                'warning' => 'Aviso',
                'info' => 'Info'
            );

            fputcsv($output, array(
                $log->timestamp,
                $log->action,
                $status_map[$log->level] ?? $log->level,
                $log->post_title ?? '',
                $log->post_slug ?? '',
                $log->post_date ?? '',
                $valid_tags,
                $log->message
            ), ';');
        }

        fclose($output);
        exit;
    }

    /**
     * Verificar se log está habilitado
     */
    private function is_logging_enabled()
    {
        return isset($this->settings['log_errors']) && $this->settings['log_errors'];
    }
}
