<?php

/**
 * Classe para validação e registro de posts
 */

if (!defined('ABSPATH')) {
    exit;
}

class WPP_ADI_Feed_Post_Validation
{

    private $api_client;
    private $settings;
    private $logger;

    public function __construct($api_client, $logger = null)
    {
        $this->api_client = $api_client;
        $this->settings = get_option('wpp_adi_feed_settings', array());
        $this->logger = $logger;
    }

    /**
     * Validar e registrar post com logs detalhados
     */
    public function validate_and_register_post($post_id, $post, $update = false)
    {
        // Debug: log sempre que o método é chamado
        error_log("WPP ADI Feed: validate_and_register_post chamado - Post ID: $post_id, Update: " . ($update ? 'true' : 'false'));

        // Verificar se deve processar
        if (!$this->should_process_post($post_id, $post, $update)) {
            error_log("WPP ADI Feed: Post $post_id não deve ser processado");
            return;
        }

        error_log("WPP ADI Feed: Processando post $post_id");

        // Verificar se auto-registro está ativado
        if (!isset($this->settings['auto_register']) || !$this->settings['auto_register']) {
            error_log("WPP ADI Feed: Auto-registro desativado");
            return;
        }

        // Obter tags válidas
        $valid_tags = $this->api_client->get_valid_tags();
        if ($valid_tags === false) {
            $this->log_error("Não foi possível obter tags válidas para o post ID {$post_id}");
            return;
        }

        // Obter tags/categorias válidas do post
        $post_valid_tags = $this->get_post_valid_tags($post_id, $valid_tags);

        // Log da validação
        if ($this->logger) {
            $this->logger->log_tag_validation($post_id, $post_valid_tags, !empty($post_valid_tags));
        }

        // Verificar se post tem tags/categorias válidas
        if (empty($post_valid_tags)) {
            return;
        }

        // Preparar dados do post
        $valid_tags_found = implode(', ', $post_valid_tags);
        $post_data = $this->api_client->prepare_post_data($post_id, $valid_tags_found);

        if (!$post_data) {
            $this->log_error("Erro ao preparar dados do post ID {$post_id}");
            return;
        }

        // Registrar na API
        $success = $this->api_client->register_post($post_data);

        if ($success) {
            error_log("WPP ADI Feed: CONFIRMAÇÃO - Post ID {$post_id} foi enviado com SUCESSO para a API");
        } else {
            error_log("WPP ADI Feed: CONFIRMAÇÃO - Post ID {$post_id} FALHOU ao ser enviado para a API");
        }

        // Log do registro
        if ($this->logger) {
            $this->logger->log_post_registration($post_id, $success, $post_valid_tags);
        }

        if ($success) {
            // Salvar flag de que foi registrado
            update_post_meta($post_id, '_wpp_adi_feed_registered', current_time('mysql'));
            update_post_meta($post_id, '_wpp_adi_feed_valid_tags', $post_valid_tags);
        }
    }


    /**
     * Verificar se post deve ser processado (sem alterações)
     */
    private function should_process_post($post_id, $post, $update = false)
    {
        error_log("WPP ADI Feed: should_process_post - Post ID: $post_id, Update: " . ($update ? 'true' : 'false'));

        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            error_log("WPP ADI Feed: Autosave detectado, pulando");
            return false;
        }

        if (wp_is_post_revision($post_id)) {
            error_log("WPP ADI Feed: É uma revisão, pulando");
            return false;
        }

        // Apenas processar posts publicados
        if ($post->post_status !== 'publish') {
            error_log("WPP ADI Feed: Status não é publish, pulando");
            return false;
        }

        $allowed_post_types = array('post', 'page');
        if (!in_array($post->post_type, $allowed_post_types)) {
            error_log("WPP ADI Feed: Tipo de post não permitido: {$post->post_type}");
            return false;
        }

        // NOVA LÓGICA: Sempre processar posts atualizados para permitir registro tardio
        // Apenas pular se for um novo post que já foi registrado (improvável)
        $already_registered = get_post_meta($post_id, '_wpp_adi_feed_registered', true);
        error_log("WPP ADI Feed: Já registrado: " . ($already_registered ? 'sim' : 'não'));

        // Se é uma atualização, sempre processar (pode ter adicionado tags válidas)
        if ($update) {
            error_log("WPP ADI Feed: É atualização, processando");
            return true;
        }

        // Para novos posts, verificar se já foi registrado (evita duplicação)
        if ($already_registered) {
            error_log("WPP ADI Feed: Novo post já registrado, pulando");
            return false;
        }

        error_log("WPP ADI Feed: Novo post não registrado, processando");
        return true;
    }

    /**
     * Obter tags/categorias válidas do post
     */
    public function get_post_valid_tags($post_id, $valid_tags = null)
    {

        if ($valid_tags === null) {
            $valid_tags = $this->api_client->get_valid_tags();
            if ($valid_tags === false) {
                return array();
            }
        }

        $post_valid_tags = array();

        // Debug: Verificar tags do post
        $post_tags = wp_get_post_tags($post_id, array('fields' => 'slugs'));

        // Também pegar tags com nomes para comparação
        $post_tags_full = wp_get_post_tags($post_id);

        foreach ($post_tags as $tag_slug) {
            if (in_array($tag_slug, $valid_tags)) {
                $post_valid_tags[] = $tag_slug;
            }
        }

        // Debug: Verificar categorias do post
        $post_categories = wp_get_post_categories($post_id, array('fields' => 'slugs'));

        // Também pegar categorias com nomes
        $post_categories_full = get_the_category($post_id);

        foreach ($post_categories as $category_slug) {
            if (in_array($category_slug, $valid_tags)) {
                $post_valid_tags[] = $category_slug;
            }
        }

        $result = array_unique($post_valid_tags);

        return $result;
    }

    /**
     * Forçar registro manual de um post
     */
    public function force_register_post($post_id)
    {
        $post = get_post($post_id);
        if (!$post || $post->post_status !== 'publish') {
            return false;
        }

        // Remover flag de já registrado
        delete_post_meta($post_id, '_wpp_adi_feed_registered');

        // Processar registro
        $this->validate_and_register_post($post_id, $post);

        // Verificar se foi registrado
        $registered = get_post_meta($post_id, '_wpp_adi_feed_registered', true);
        return !empty($registered);
    }

    /**
     * Verificar se post foi registrado
     */
    public function is_post_registered($post_id)
    {
        $registered = get_post_meta($post_id, '_wpp_adi_feed_registered', true);
        return !empty($registered);
    }

    /**
     * Log de erros (mantido para compatibilidade)
     */
    private function log_error($message)
    {
        if (isset($this->settings['log_errors']) && $this->settings['log_errors']) {
            // Log de erro removido
        }
    }

    /**
     * Log de informações (mantido para compatibilidade)
     */
    private function log_info($message)
    {
        if (isset($this->settings['log_errors']) && $this->settings['log_errors']) {
            // Log de info removido
        }
    }
}
