<?php

/**
 * Sistema de Atualização Automática para WPP ADI Feed
 * Classe responsável por verificar e gerenciar atualizações do plugin
 * 
 * Arquivo: includes/class-updater.php
 */

if (!defined('ABSPATH')) {
    exit;
}

class WPP_ADI_Feed_Updater
{

    private $plugin_slug;
    private $plugin_basename;
    private $version;
    private $cache_key;
    private $cache_allowed;
    private $update_endpoint;

    public function __construct($plugin_file, $version)
    {
        $this->plugin_basename = plugin_basename($plugin_file);
        $this->plugin_slug = dirname($this->plugin_basename);
        $this->version = $version;
        $this->cache_key = 'wpp_adi_feed_update_check';
        $this->cache_allowed = true;
        $this->update_endpoint = 'https://relatorios.prportais.com.br/feed-wp-adi-plugin-update';
        add_filter('pre_set_site_transient_update_plugins', array($this, 'check_for_update'));
        add_filter('plugins_api', array($this, 'plugin_popup'), 10, 3);
        add_filter('upgrader_pre_download', array($this, 'upgrader_pre_download'), 10, 3);

        // Limpar cache quando plugin for ativado/desativado
        add_action('activated_plugin', array($this, 'clear_update_cache'));
        add_action('deactivated_plugin', array($this, 'clear_update_cache'));

        // Adicionar ação para verificação manual
        add_action('admin_post_wpp_adi_feed_check_update', array($this, 'manual_update_check'));

        // Adicionar botão de verificação manual na página do plugin
        add_filter('plugin_row_meta', array($this, 'plugin_row_meta'), 10, 2);

        // Admin notices para novas versões
        add_action('admin_notices', array($this, 'update_notice'));
    }

    /**
     * Verificar se há atualizações disponíveis
     */
    public function check_for_update($checked_data)
    {
        if (empty($checked_data->checked)) {
            return $checked_data;
        }

        $remote_version = $this->get_remote_version();

        if (version_compare($this->version, $remote_version['version'], '<')) {
            $checked_data->response[$this->plugin_basename] = (object) array(
                'slug' => $this->plugin_slug,
                'plugin' => $this->plugin_basename,
                'new_version' => $remote_version['version'],
                'tested' => $remote_version['tested'],
                'package' => $remote_version['download_url'],
                'upgrade_notice' => $remote_version['upgrade_notice']
            );
        }

        return $checked_data;
    }

    /**
     * Obter informações da versão remota
     */
    private function get_remote_version()
    {
        $cache_key = $this->cache_key . '_data';

        if ($this->cache_allowed && false !== ($cached = get_transient($cache_key))) {
            return $cached;
        }

        $request = wp_remote_get($this->update_endpoint, array(
            'timeout' => 30,
            'headers' => array(
                'Accept' => 'application/json',
                'User-Agent' => 'WPP-ADI-Feed-Updater/' . $this->version . '; ' . get_bloginfo('url')
            )
        ));

        if (!is_wp_error($request) && wp_remote_retrieve_response_code($request) === 200) {
            $body = wp_remote_retrieve_body($request);
            $data = json_decode($body, true);

            if (is_array($data) && isset($data['version'])) {
                $remote_version = array(
                    'version' => $data['version'],
                    'download_url' => $data['download_url'],
                    'requires' => $data['requires'] ?? '5.0',
                    'tested' => $data['tested'] ?? '6.4',
                    'author' => $data['author'] ?? 'Trivium Dev Br',
                    'homepage' => $data['homepage'] ?? 'https://trivium.dev.br',
                    'upgrade_notice' => $data['upgrade_notice'] ?? ''
                );

                // Cache por 12 horas
                if ($this->cache_allowed) {
                    set_transient($cache_key, $remote_version, 12 * HOUR_IN_SECONDS);
                }

                return $remote_version;
            }
        }

        // Retornar versão atual se não conseguir verificar
        return array(
            'version' => $this->version,
            'download_url' => '',
            'requires' => '5.0',
            'tested' => '6.4',
            'author' => 'Trivium Dev Br',
            'homepage' => 'https://trivium.dev.br',
            'upgrade_notice' => ''
        );
    }

    /**
     * Popup de informações do plugin
     */
    public function plugin_popup($result, $action, $args)
    {
        if ($action !== 'plugin_information') {
            return $result;
        }

        if (!isset($args->slug) || $args->slug !== $this->plugin_slug) {
            return $result;
        }

        $remote_version = $this->get_remote_version();

        return (object) array(
            'name' => 'WPP ADI Feed',
            'slug' => $this->plugin_slug,
            'version' => $remote_version['version'],
            'author' => $remote_version['author'],
            'homepage' => $remote_version['homepage'],
            'requires' => $remote_version['requires'],
            'tested' => $remote_version['tested'],
            'download_link' => $remote_version['download_url'],
            'sections' => array(
                'description' => 'Plugin que integra WordPress com API de relatórios, validando tags e registrando posts automaticamente.',
                'changelog' => $remote_version['upgrade_notice']
            )
        );
    }

    /**
     * Pre-download para autenticação se necessária
     */
    public function upgrader_pre_download($reply, $package, $upgrader)
    {
        if (strpos($package, 'relatorios.prportais.com.br') !== false) {
            // Aqui você pode adicionar autenticação se necessária
            return $reply;
        }

        return $reply;
    }

    /**
     * Forçar verificação de atualização
     */
    public function force_update_check()
    {
        $this->cache_allowed = false;
        $this->clear_update_cache();

        // Forçar WordPress a verificar atualizações
        wp_update_plugins();
    }

    /**
     * Verificação manual de atualização
     */
    public function manual_update_check()
    {
        if (!current_user_can('manage_options')) {
            wp_die(__('Acesso negado', 'wpp-adi-feed'));
        }

        if (!wp_verify_nonce($_GET['_wpnonce'], 'wpp_adi_feed_check_update')) {
            wp_die(__('Nonce inválido', 'wpp-adi-feed'));
        }

        $this->force_update_check();

        $message = 'update_checked';
        $redirect_url = admin_url('plugins.php?message=' . $message);
        wp_redirect($redirect_url);
        exit;
    }

    /**
     * Limpar cache de atualizações
     */
    public function clear_update_cache()
    {
        delete_transient($this->cache_key . '_data');
        delete_site_transient('update_plugins');
    }

    /**
     * Adicionar link de verificação manual na lista de plugins
     */
    public function plugin_row_meta($links, $file)
    {
        if ($file === $this->plugin_basename) {
            $check_update_link = wp_nonce_url(
                admin_url('admin-post.php?action=wpp_adi_feed_check_update'),
                'wpp_adi_feed_check_update'
            );

            $links[] = '<a href="' . $check_update_link . '">' . __('Verificar Atualização', 'wpp-adi-feed') . '</a>';
        }

        return $links;
    }

    /**
     * Mostrar notice de nova versão disponível
     */
    public function update_notice()
    {
        if (!current_user_can('update_plugins')) {
            return;
        }

        $remote_version = $this->get_remote_version();

        if (version_compare($this->version, $remote_version['version'], '<')) {
            $plugin_data = get_plugin_data(WPP_ADI_FEED_PLUGIN_DIR . 'wpp-adi-feed.php');
            $update_url = wp_nonce_url(
                self_admin_url('update.php?action=upgrade-plugin&plugin=' . urlencode($this->plugin_basename)),
                'upgrade-plugin_' . $this->plugin_basename
            );

            echo '<div class="notice notice-warning is-dismissible">';
            echo '<p><strong>' . $plugin_data['Name'] . '</strong></p>';
            echo '<p>' . sprintf(
                __('Nova versão disponível: %s (você tem %s). %s', 'wpp-adi-feed'),
                '<strong>' . $remote_version['version'] . '</strong>',
                $this->version,
                '<a href="' . $update_url . '" class="update-link">Atualizar agora</a>'
            ) . '</p>';

            if (!empty($remote_version['upgrade_notice'])) {
                echo '<p><em>' . esc_html($remote_version['upgrade_notice']) . '</em></p>';
            }
            echo '</div>';
        }
    }

    /**
     * Obter status da última verificação
     */
    public function get_update_status()
    {
        $remote_version = $this->get_remote_version();
        $has_update = version_compare($this->version, $remote_version['version'], '<');
        $cache_data = get_transient($this->cache_key . '_data');

        return array(
            'current_version' => $this->version,
            'latest_version' => $remote_version['version'],
            'has_update' => $has_update,
            'last_check' => $cache_data ? get_transient('_transient_timeout_' . $this->cache_key . '_data') : false,
            'update_url' => wp_nonce_url(
                self_admin_url('update.php?action=upgrade-plugin&plugin=' . urlencode($this->plugin_basename)),
                'upgrade-plugin_' . $this->plugin_basename
            )
        );
    }

    /**
     * Adicionar informações de atualização à página de configurações
     */
    public function add_update_info_to_settings()
    {
        $status = $this->get_update_status();

?>
        <hr>
        <h2><?php _e('Informações de Atualização', 'wpp-adi-feed'); ?></h2>
        <table class="form-table">
            <tr>
                <th><?php _e('Versão Atual', 'wpp-adi-feed'); ?></th>
                <td><?php echo esc_html($status['current_version']); ?></td>
            </tr>
            <tr>
                <th><?php _e('Última Versão', 'wpp-adi-feed'); ?></th>
                <td>
                    <?php echo esc_html($status['latest_version']); ?>
                    <?php if ($status['has_update']): ?>
                        <span style="color: orange; font-weight: bold;">
                            (<?php _e('Atualização disponível!', 'wpp-adi-feed'); ?>)
                        </span>
                    <?php else: ?>
                        <span style="color: green;">
                            ✓ <?php _e('Atualizado', 'wpp-adi-feed'); ?>
                        </span>
                    <?php endif; ?>
                </td>
            </tr>
            <tr>
                <th><?php _e('Última Verificação', 'wpp-adi-feed'); ?></th>
                <td>
                    <?php
                    if ($status['last_check']) {
                        echo esc_html(date_i18n('d/m/Y H:i:s', $status['last_check']));
                    } else {
                        echo '<em>' . __('Nunca verificado', 'wpp-adi-feed') . '</em>';
                    }
                    ?>
                </td>
            </tr>
        </table>

        <?php if ($status['has_update']): ?>
            <p>
                <a href="<?php echo esc_url($status['update_url']); ?>" class="button button-primary">
                    <?php _e('Atualizar Agora', 'wpp-adi-feed'); ?>
                </a>
                <a href="<?php echo wp_nonce_url(admin_url('admin-post.php?action=wpp_adi_feed_check_update'), 'wpp_adi_feed_check_update'); ?>" class="button button-secondary">
                    <?php _e('Verificar Novamente', 'wpp-adi-feed'); ?>
                </a>
            </p>
        <?php else: ?>
            <p>
                <a href="<?php echo wp_nonce_url(admin_url('admin-post.php?action=wpp_adi_feed_check_update'), 'wpp_adi_feed_check_update'); ?>" class="button button-secondary">
                    <?php _e('Verificar Atualizações', 'wpp-adi-feed'); ?>
                </a>
            </p>
        <?php endif; ?>
<?php
    }



    /**
     * Envia um relatório à API quando o plugin é ativado, desativado ou atualizado.
     *
     * @param string $action Ação realizada: 'activated', 'deactivated', 'updated'
     */
    public function send_activation_deactivation_report($action)
    {
        // Para atualizações, ler a versão diretamente do cabeçalho do arquivo (sem cache)
        $version = WPP_ADI_FEED_VERSION;
        if ($action === 'updated') {
            $plugin_file = WP_PLUGIN_DIR . '/wpp-adi-feed/wpp-adi-feed.php';
            if (file_exists($plugin_file)) {
                // Limpar cache do OPcache se disponível
                if (function_exists('opcache_invalidate')) {
                    opcache_invalidate($plugin_file, true);
                }
                
                // Ler versão diretamente do arquivo
                $file_contents = file_get_contents($plugin_file);
                if (preg_match('/\*\s*Version:\s*(.+)$/mi', $file_contents, $matches)) {
                    $version = trim($matches[1]);
                } else {
                    // Fallback: usar get_plugin_data
                    $plugin_data = get_plugin_data($plugin_file, false, false);
                    $version = $plugin_data['Version'] ?? WPP_ADI_FEED_VERSION;
                }
            }
        }

        $is_active = in_array($action, ['activated', 'updated']) ? 1 : 0;

        $data = [
            'domain' => esc_url(home_url()),
            'wp_active' => $is_active,
            'version' => $version,
            'action' => $action // Adiciona o tipo de ação
        ];

        $response = wp_remote_post('https://relatorios.prportais.com.br/feed-wp-ativar', [
            'method' => 'POST',
            'timeout' => 15,
            'headers' => [
                'Content-Type' => 'application/json',
            ],
            'body' => json_encode($data),
        ]);

        // Verificar se houve erro na requisição
        if (is_wp_error($response)) {
            error_log('Erro ao enviar relatório de ' . $action . ': ' . $response->get_error_message());
        } else {
            $response_code = wp_remote_retrieve_response_code($response);
            if ($response_code !== 200) {
                error_log('Erro ao enviar relatório de ' . $action . '. Código de resposta: ' . $response_code);
            }
        }
    }

    /**
     * Envia um relatório à API quando o plugin é atualizado.
     * @deprecated Use send_activation_deactivation_report('updated') instead
     */
    public function send_update_report()
    {
        $this->send_activation_deactivation_report('updated');
    }
}
