<?php

/**
 * Plugin Name: WPP ADI Feed
 * Description: Plugin que integra WordPress com API de relatórios, validando tags e registrando posts automaticamente
 * Version: 2.2.12
 * Author: Lindomar | Trirvium Dev BR
 * Text Domain: wpp-adi-feed
 * Requires at least: 6.0
 */

// Prevenir acesso direto
if (!defined('ABSPATH')) {
    exit;
}

// Definir constantes
define('WPP_ADI_FEED_VERSION', '2.2.12');
define('WPP_ADI_FEED_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('WPP_ADI_FEED_PLUGIN_URL', plugin_dir_url(__FILE__));
define('WPP_ADI_FEED_PLUGIN_FILE', __FILE__);

// Bloquear instalação/execução em versões do WordPress menores que 6.0
if (version_compare(get_bloginfo('version'), '6.0', '<')) {
    add_action('admin_notices', function () {
        echo '<div class="notice notice-error"><p><strong>WPP ADI Feed:</strong> Este plugin requer o WordPress 6.0 ou superior.</p></div>';
    });
    // Impede execução do restante do plugin
    return;
}

/**
 * Função para obter a URL base da API das configurações
 */
function wpp_adi_feed_get_api_base_url()
{
    $settings = get_option('wpp_adi_feed_settings', array());
    return isset($settings['api_base_url']) ? $settings['api_base_url'] : 'https://relatorios.prportais.com.br/api';
}

/**
 * Função para obter a URL do script CDN das configurações
 */
function wpp_adi_feed_get_cdn_script_url()
{
    $settings = get_option('wpp_adi_feed_settings', array());
    return isset($settings['cdn_script_url']) ? $settings['cdn_script_url'] : 'https://relatorios.prportais.com.br/storage/cdn-adi/news-tracker-wp-min.js';
}

// Verificar se os arquivos existem antes de incluir
$required_files = array(
    'includes/class-updater.php',
    'includes/class-api-client.php',
    'includes/class-post-validation.php',
    'includes/class-admin-settings.php',
    'includes/class-log-admin.php',
    'includes/class-logger.php',
    'includes/legado-wpp-adi-feed.php',
);

foreach ($required_files as $file) {
    $file_path = WPP_ADI_FEED_PLUGIN_DIR . $file;
    if (!file_exists($file_path)) {
        add_action('admin_notices', function () use ($file) {
            echo '<div class="notice notice-error"><p>WPP ADI Feed: Arquivo obrigatório não encontrado: ' . esc_html($file) . '</p></div>';
        });
        return;
    }
    require_once $file_path;
}

class WPP_ADI_Feed
{
    /**
     * @var WPP_ADI_Feed_Updater
     */
    public $updater;
    private $api_client;
    private $post_validation;
    private $admin_settings;
    private $logger;
    private $log_admin;

    public function __construct()
    {
        $this->init();

        // Registrar o hook para limpeza diária de logs
        add_action('wpp_adi_feed_daily_log_cleanup', array($this, 'daily_log_cleanup'));

        // Verificar status do schedule
        add_action('admin_init', array($this, 'check_schedule_status'));

        // Registrar ação para ativar o schedule manualmente
        add_action('admin_post_activate_daily_log_cleanup', array($this, 'activate_daily_log_cleanup'));

        // Handler para limpeza manual de logs
        add_action('admin_post_wpp_adi_feed_manual_log_cleanup', array($this, 'manual_log_cleanup_handler'));

        // Agendar evento ao atualizar o plugin
        add_action('upgrader_process_complete', function ($upgrader_object, $options) {
            if ($options['action'] === 'update' && $options['type'] === 'plugin') {
                $current_plugin_path = plugin_basename(__FILE__);
                if (in_array($current_plugin_path, $options['plugins'])) {
                    $this->schedule_daily_log_cleanup();
                    // Notificar API sobre a atualização
                    $this->updater->send_activation_deactivation_report('updated');
                }
            }
        }, 10, 2);
    }

    /**
     * Handler para limpeza manual de logs via painel
     */
    public function manual_log_cleanup_handler()
    {
        // Verificar permissões
        if (!current_user_can('manage_options')) {
            wp_die(__('Você não tem permissão para executar esta ação.', 'wpp-adi-feed'));
        }

        // Verificar nonce (aceita tanto POST quanto GET)
        $nonce_value = isset($_POST['wpp_adi_feed_nonce']) ? $_POST['wpp_adi_feed_nonce'] : (isset($_GET['wpp_adi_feed_nonce']) ? $_GET['wpp_adi_feed_nonce'] : '');
        if (!wp_verify_nonce($nonce_value, 'wpp_adi_feed_manual_log_cleanup')) {
            wp_die(__('Verificação de segurança falhou.', 'wpp-adi-feed'));
        }

        // Obter configurações
        $settings = get_option('wpp_adi_feed_settings', array());
        $days_to_keep = isset($settings['log_retention_days']) ? intval($settings['log_retention_days']) : 7;

        $deleted_count = 0;

        // Executar limpeza se logger estiver disponível
        if ($this->logger) {
            $deleted_count = $this->logger->cleanup_old_logs($days_to_keep);
        }

        // Redirecionar de volta com mensagem de sucesso
        $redirect_url = add_query_arg(array(
            'page' => 'wpp-adi-feed',
            'message' => 'manual_log_cleanup_done',
            'deleted' => $deleted_count
        ), admin_url('options-general.php'));

        wp_safe_redirect($redirect_url);
        exit;
    }

    private function init()
    {
        // Verificar se as classes existem
        if (
            !class_exists('WPP_ADI_Feed_API_Client') ||
            !class_exists('WPP_ADI_Feed_Post_Validation') ||
            !class_exists('WPP_ADI_Feed_Admin_Settings') ||
            !class_exists('WPP_ADI_Feed_Logger')  ||
            !class_exists('WPP_ADI_Feed_Updater')
        ) {
            add_action('admin_notices', array($this, 'missing_classes_notice'));
            return;
        }
        // Inicializar sistema de atualizações primeiro
        $this->updater = new WPP_ADI_Feed_Updater(WPP_ADI_FEED_PLUGIN_FILE, WPP_ADI_FEED_VERSION);

        // Inicializar logger primeiro
        $this->logger = new WPP_ADI_Feed_Logger();

        // Inicializar classes com logger
        $this->api_client = new WPP_ADI_Feed_API_Client($this->logger);
        $this->post_validation = new WPP_ADI_Feed_Post_Validation($this->api_client, $this->logger);
        $this->admin_settings = new WPP_ADI_Feed_Admin_Settings();
        $this->log_admin = new WPP_ADI_Feed_Log_Admin($this->logger);

        // Hooks
        add_action('init', array($this, 'load_textdomain'));
        add_action('admin_enqueue_scripts', array($this, 'admin_scripts'));

        add_action('wp_after_insert_post', array($this, 'handle_after_insert_post'), 10, 4);
        add_action('rest_after_insert_post', array($this, 'handle_rest_after_insert'), 10, 3);
        add_action('future_to_publish', array($this, 'handle_scheduled_post'), 10, 1);

        // Adicionar ação para o evento agendado:
        add_action('wpp_adi_process_post_delayed', array($this, 'process_delayed_post'));

        add_action('wp_ajax_load_gateway_tags', array($this, 'ajax_load_gateway_tags'));
        add_action('wp_ajax_validate_current_post_tags', array($this, 'ajax_validate_current_post_tags'));

        // Hook para validação inicial quando editar/criar post
        add_action('admin_init', array($this, 'init_post_editor_validation'));

        // Agendar limpeza automática de logs antigos
        if (!wp_next_scheduled('wpp_adi_feed_cleanup_logs')) {
            wp_schedule_event(time(), 'weekly', 'wpp_adi_feed_cleanup_logs');
        }
        add_action('wpp_adi_feed_cleanup_logs', array($this, 'scheduled_log_cleanup'));

        // Agendar limpeza diária de logs
        // $this->schedule_daily_log_cleanup(); // Removido para controle manual via painel

        // Ativação/Desativação
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }

    private function schedule_daily_log_cleanup()
    {
        if (!wp_next_scheduled('wpp_adi_feed_daily_log_cleanup')) {
            wp_schedule_event(strtotime('01:00:00'), 'daily', 'wpp_adi_feed_daily_log_cleanup');
        }
    }

    private function unschedule_daily_log_cleanup()
    {
        $timestamp = wp_next_scheduled('wpp_adi_feed_daily_log_cleanup');
        if ($timestamp) {
            wp_unschedule_event($timestamp, 'wpp_adi_feed_daily_log_cleanup');
        }
    }

    public function daily_log_cleanup()
    {
        if ($this->logger) {
            $settings = get_option('wpp_adi_feed_settings', array());
            $days_to_keep = isset($settings['log_retention_days']) ? intval($settings['log_retention_days']) : 7;
            $this->logger->cleanup_old_logs($days_to_keep);
        }
    }

    public function check_schedule_status()
    {
        if (!wp_next_scheduled('wpp_adi_feed_daily_log_cleanup')) {
            add_action('admin_notices', function () {
                echo '<div class="notice notice-warning"><p><strong>WPP ADI Feed:</strong> O evento de limpeza diária de logs não está ativo. <a href="' . esc_url(admin_url('admin-post.php?action=activate_daily_log_cleanup')) . '">Clique aqui para ativá-lo</a>.</p></div>';
            });
        }
    }

    public function activate_daily_log_cleanup()
    {
        $this->schedule_daily_log_cleanup();
        wp_redirect(admin_url());
        exit;
    }

    public function init_post_editor_validation()
    {
        global $pagenow, $typenow;

        // Apenas nas páginas de edição de posts
        if (in_array($pagenow, array('post.php', 'post-new.php')) && in_array($typenow, array('post', 'page'))) {
            // Adicionar script para validação inicial
            add_action('admin_footer', array($this, 'post_editor_validation_script'));
        }
    }

    public function post_editor_validation_script()
    {
?>
        <script type="text/javascript">
            jQuery(document).ready(function($) {
                // Aguardar o editor carregar completamente
                setTimeout(function() {
                    if (typeof initWppAdiFeed === 'function') {
                        // Forçar validação inicial
                        loadValidTags();
                    }
                }, 2000);
            });
        </script>
<?php
    }


    public function handle_after_insert_post($post_id, $post, $update, $post_before)
    {
        if ($post && $post->post_status === 'publish') {
            // Processar imediatamente (mais confiável)
            $this->post_validation->validate_and_register_post($post_id, $post, $update);
        }
    }

    public function handle_rest_after_insert($post, $request, $creating)
    {
        if ($post->post_status === 'publish') {
            $this->post_validation->validate_and_register_post($post->ID, $post);
        }
    }

    public function process_delayed_post($post_id)
    {
        $post = get_post($post_id);
        if ($post && $post->post_status === 'publish') {
            $this->post_validation->validate_and_register_post($post_id, $post);
        }
    }

    public function handle_scheduled_post($post)
    {
        if (is_object($post)) {
            $this->post_validation->validate_and_register_post($post->ID, $post, false);
        }
    }

    /**
     * Limpeza agendada de logs antigos (executada semanalmente)
     */
    public function scheduled_log_cleanup()
    {
        if ($this->logger) {
            $settings = get_option('wpp_adi_feed_settings', array());
            $days_to_keep = isset($settings['log_retention_days']) ? intval($settings['log_retention_days']) : 7;
            $this->logger->cleanup_old_logs($days_to_keep);
        }
    }

    public function missing_classes_notice()
    {
        echo '<div class="notice notice-error"><p>WPP ADI Feed: Erro ao carregar classes do plugin. Verifique se todos os arquivos estão presentes.</p></div>';
    }

    public function load_textdomain()
    {
        load_plugin_textdomain('wpp-adi-feed', false, dirname(plugin_basename(__FILE__)) . '/languages/');
    }

    public function admin_scripts($hook)
    {
        // Carregar apenas nas páginas de edição ou adição de posts
        if (!in_array($hook, array('post.php', 'post-new.php'))) {
            return;
        }

        wp_enqueue_script(
            'wpp-adi-feed-admin',
            WPP_ADI_FEED_PLUGIN_URL . 'assets/js/admin.js',
            array('jquery'),
            WPP_ADI_FEED_VERSION,
            true
        );

        wp_enqueue_style(
            'wpp-adi-feed-admin',
            WPP_ADI_FEED_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            WPP_ADI_FEED_VERSION
        );

        // Localizar script
        wp_localize_script('wpp-adi-feed-admin', 'wppAdiFeed', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('wpp_adi_feed_nonce'),
            'loading_text' => __('Carregando tags válidas...', 'wpp-adi-feed'),
            'api_base_url' => wpp_adi_feed_get_api_base_url(),
            'cdn_script_url' => wpp_adi_feed_get_cdn_script_url()
        ));
    }

    public function ajax_load_gateway_tags()
    {
        check_ajax_referer('wpp_adi_feed_nonce', 'nonce');

        if (!$this->api_client) {
            wp_send_json_error(__('Cliente API não inicializado', 'wpp-adi-feed'));
            return;
        }

        $tags = $this->api_client->get_valid_tags();

        if ($tags !== false) {
            wp_send_json_success($tags);
        } else {
            wp_send_json_error(__('Erro ao carregar tags da API', 'wpp-adi-feed'));
        }
    }

    /**
     * AJAX para validar tags do post atual em tempo real
     */
    public function ajax_validate_current_post_tags()
    {
        check_ajax_referer('wpp_adi_feed_nonce', 'nonce');

        $post_id = intval($_POST['post_id'] ?? 0);
        $tag_ids = array_map('intval', $_POST['tag_ids'] ?? array());
        $category_ids = array_map('intval', $_POST['category_ids'] ?? array());

        if (!$post_id || (!$tag_ids && !$category_ids)) {
            wp_send_json_error(__('Dados insuficientes para validação', 'wpp-adi-feed'));
            return;
        }

        // Obter tags válidas da API
        $valid_tags = $this->api_client->get_valid_tags();
        if ($valid_tags === false) {
            wp_send_json_error(__('Erro ao carregar tags válidas da API', 'wpp-adi-feed'));
            return;
        }

        $post_valid_tags = array();

        // Verificar tags
        foreach ($tag_ids as $tag_id) {
            $tag = get_term($tag_id, 'post_tag');
            if ($tag && !is_wp_error($tag) && in_array($tag->slug, $valid_tags)) {
                $post_valid_tags[] = $tag->slug;
            }
        }

        // Verificar categorias
        foreach ($category_ids as $category_id) {
            $category = get_term($category_id, 'category');
            if ($category && !is_wp_error($category) && in_array($category->slug, $valid_tags)) {
                $post_valid_tags[] = $category->slug;
            }
        }

        wp_send_json_success(array(
            'valid_tags' => array_unique($post_valid_tags),
            'will_register' => !empty($post_valid_tags)
        ));
    }

    public function activate()
    {
        // Configurações padrão com URLs centralizadas
        $default_settings = array(
            'api_base_url' => wpp_adi_feed_get_api_base_url(),
            'cdn_script_url' => 'https://relatorios.prportais.com.br/storage/cdn-adi/news-tracker-wp-min.js',
            'request_timeout' => 10,
            'cache_duration' => 3600,
            'auto_register' => true,
            'log_errors' => true
        );

        add_option('wpp_adi_feed_settings', $default_settings);

        // Limpar cache
        delete_transient('wpp_adi_feed_valid_tags');

        // Verificar se pasta assets existe
        $assets_dir = WPP_ADI_FEED_PLUGIN_DIR . 'assets';
        if (!is_dir($assets_dir)) {
            wp_mkdir_p($assets_dir . '/js');
            wp_mkdir_p($assets_dir . '/css');
        }

        // Agendar limpeza diária de logs
        // $this->schedule_daily_log_cleanup(); // Removido para controle manual via painel

        // Enviar requisição POST para a API de relatórios ao ativar o plugin
        $this->updater->send_activation_deactivation_report('activated');
    }

    public function deactivate()
    {
        // Limpar cache
        delete_transient('wpp_adi_feed_valid_tags');

        // Limpar eventos agendados
        wp_clear_scheduled_hook('wpp_adi_feed_cleanup_logs');
        wp_clear_scheduled_hook('wpp_adi_feed_daily_log_cleanup');

        // Enviar requisição POST para a API de relatórios ao desativar o plugin
        $this->updater->send_activation_deactivation_report('deactivated');
    }

    public function update()
    {
        // Atualizar configurações ou realizar migrações, se necessário
        $this->schedule_daily_log_cleanup();
    }
}

// Função para adicionar script CDN aos posts usando URL global
function wpp_adi_feed_add_script_to_posts()
{
    // Verificar se é um post individual do tipo 'post'
    if (is_single() && is_singular('post')) {
        $cdn_script_url = wpp_adi_feed_get_cdn_script_url();

        // Registrar e enfileirar o script
        wp_enqueue_script(
            'wpp-adi-feed-script', // Identificador único
            $cdn_script_url, // URL do script obtida das configurações
            [], // Dependências (nenhuma neste caso)
            '1.2.0', // Versão do script
            false // Carregar no <head> (true = footer)
        );
    }
}
add_action('wp_enqueue_scripts', 'wpp_adi_feed_add_script_to_posts');

// Adicionar atributos async ao script (separado da função principal)
function wpp_adi_feed_add_async_attribute($tag, $handle)
{
    if ('wpp-adi-feed-script' === $handle) {
        return str_replace(' src', ' async="async" src', $tag);
    }
    return $tag;
}
add_filter('script_loader_tag', 'wpp_adi_feed_add_async_attribute', 10, 2);

// Forçar autoupdate para este plugin
add_filter('auto_update_plugin', function ($update, $item) {
    if (isset($item->slug) && $item->slug === 'wpp-adi-feed') {
        return true;
    }
    // Alternativamente, pode-se usar plugin_basename(__FILE__) para garantir compatibilidade
    if (isset($item->plugin) && strpos($item->plugin, 'wpp-adi-feed.php') !== false) {
        return true;
    }
    return $update;
}, 10, 2);

// Inicializar plugin apenas se não estiver em processo de desativação
if (!defined('WP_UNINSTALL_PLUGIN')) {
    new WPP_ADI_Feed();
}
